<?php

namespace App\Filament\Resources\Membership\MembershipPlanResource\Pages;

use App\Filament\Resources\Membership\MembershipPlanResource;
use Filament\Resources\Pages\ViewRecord;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Components\Grid;
use Filament\Infolists\Components\Group;
use Filament\Infolists\Components\IconEntry;
use Filament\Infolists\Components\Tabs;
use Filament\Infolists\Components\Tabs\Tab;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconPosition;
use Illuminate\Support\Number;
use Filament\Resources\RelationManagers\RelationGroup;
use App\Filament\Resources\Membership\MemberShipPlanResource\RelationManagers\SubscriptionsRelationManager;

class ViewMembershipPlan extends ViewRecord
{
    protected static string $resource = MembershipPlanResource::class;

    /**
     * @return array<RelationGroup | string>
     */
    public function getRelationManagers(): array
    {
        return [
            SubscriptionsRelationManager::class,
        ];
    }

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                // Hero Section with Plan Overview
                Section::make()
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                Group::make([
                                    TextEntry::make('package_title')
                                        ->label('Package Title')
                                        ->weight(FontWeight::Bold)
                                        ->size(TextEntry\TextEntrySize::Large),
                                    TextEntry::make('membership_type')
                                        ->badge()
                                        ->color('primary')
                                        ->icon('heroicon-m-bookmark')
                                        ->iconPosition(IconPosition::Before),
                                    TextEntry::make('role.name')
                                        ->label('Role')
                                        ->icon('heroicon-m-user-circle')
                                        ->color('gray'),
                                ])->columnSpan(1),
                                
                                Group::make([
                                    TextEntry::make('package_description')
                                        ->markdown()
                                        ->columnSpanFull(),
                                ])->columnSpan(1),
                            ]),
                    ])->columnSpanFull(),

                // Quick Stats Cards
                Grid::make(4)
                    ->schema([
                        Section::make()
                            ->schema([
                                TextEntry::make('subscriptions_count')
                                    ->label('Total Members')
                                    ->state(fn ($record) => $record->subscriptions->count())
                                    ->badge()
                                    ->size(TextEntry\TextEntrySize::Large)
                                    ->color('info')
                                    ->icon('heroicon-m-users'),
                            ])
                            ->columnSpan(1),

                        Section::make()
                            ->schema([
                                TextEntry::make('active_subscriptions')
                                    ->label('Active Members')
                                    ->state(fn ($record) => $record->subscriptions->where('is_active', true)->count())
                                    ->badge()
                                    ->size(TextEntry\TextEntrySize::Large)
                                    ->color('success')
                                    ->icon('heroicon-m-check-circle'),
                            ])
                            ->columnSpan(1),

                        Section::make()
                            ->schema([
                                TextEntry::make('trial_subscriptions')
                                    ->label('Trial Members')
                                    ->state(fn ($record) => $record->subscriptions->whereNotNull('trial_ends_at')->where('trial_ends_at', '>', now())->count())
                                    ->badge()
                                    ->size(TextEntry\TextEntrySize::Large)
                                    ->color('warning')
                                    ->icon('heroicon-m-clock'),
                            ])
                            ->columnSpan(1),

                        Section::make()
                            ->schema([
                                TextEntry::make('cancelled_subscriptions')
                                    ->label('Cancelled')
                                    ->state(fn ($record) => $record->subscriptions->whereNotNull('cancelled_at')->count())
                                    ->badge()
                                    ->size(TextEntry\TextEntrySize::Large)
                                    ->color('danger')
                                    ->icon('heroicon-m-x-circle'),
                            ])
                            ->columnSpan(1),
                    ])->columnSpanFull(),

                // Main Content Grid
                Grid::make(3)
                    ->schema([
                        // Left Column - Plan Details
                        Group::make([
                            Section::make('Plan Settings')
                                ->icon('heroicon-m-cog-6-tooth')
                                ->collapsible()
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            IconEntry::make('is_active')
                                                ->label('Status')
                                                ->boolean()
                                                ->color('success'),
                                            IconEntry::make('is_visible')
                                                ->label('Visibility')
                                                ->boolean()
                                                ->color('warning'),
                                            IconEntry::make('is_featured')
                                                ->label('Featured')
                                                ->boolean()
                                                ->color('info'),
                                            TextEntry::make('display_order')
                                                ->label('Display Order')
                                                ->icon('heroicon-m-arrows-up-down'),
                                        ]),
                                ]),

                            Section::make('Pricing Details')
                                ->icon('heroicon-m-banknotes')
                                ->collapsible()
                                ->schema([
                                    TextEntry::make('price_monthly')
                                        ->label('Monthly')
                                        ->money('USD')
                                        ->icon('heroicon-m-currency-dollar')
                                        ->color('success')
                                        ->size(TextEntry\TextEntrySize::Large),
                                    Grid::make(2)
                                        ->schema([
                                            TextEntry::make('price_quarterly')
                                                ->label('Quarterly')
                                                ->money('USD')
                                                ->icon('heroicon-m-currency-dollar'),
                                            TextEntry::make('price_half_yearly')
                                                ->label('Half Yearly')
                                                ->money('USD')
                                                ->icon('heroicon-m-currency-dollar'),
                                            TextEntry::make('price_yearly')
                                                ->label('Yearly')
                                                ->money('USD')
                                                ->icon('heroicon-m-currency-dollar')
                                                ->color('success'),
                                        ]),
                                ]),

                            Section::make('Trial Information')
                                ->icon('heroicon-m-gift')
                                ->collapsible()
                                ->hidden(fn ($record) => !$record->has_trial)
                                ->schema([
                                    IconEntry::make('has_trial')
                                        ->label('Trial Available')
                                        ->boolean()
                                        ->color('success'),
                                    TextEntry::make('trial_days')
                                        ->label('Trial Period')
                                        ->suffix(' days')
                                        ->icon('heroicon-m-calendar'),
                                ]),
                        ])->columnSpan(1),

                        // Middle Column - Modules
                        Group::make([
                            Section::make('Available Modules')
                                ->icon('heroicon-m-puzzle-piece')
                                ->collapsible()
                                ->schema([
                                    ...collect($this->record->planModules)->map(fn ($module) => 
                                        Section::make($module->module_name)
                                            ->icon('heroicon-m-puzzle-piece')
                                            ->collapsible()
                                            ->compact()
                                            ->schema([
                                                Grid::make(2)
                                                    ->schema([
                                                        IconEntry::make("module_allow_listing")
                                                            ->state($module->module_allow_listing)
                                                            ->label('Posting Enabled')
                                                            ->boolean()
                                                            ->color(fn ($state) => $state ? 'success' : 'danger'),
                                                        TextEntry::make("module_listing_limit")
                                                            ->state($module->module_listing_limit)
                                                            ->label('Listing Limit')
                                                            ->icon('heroicon-m-list-bullet')
                                                            ->badge(),
                                                        TextEntry::make("module_allowed_images")
                                                            ->state($module->module_allowed_images)
                                                            ->label('Images Per Listing')
                                                            ->icon('heroicon-m-photo')
                                                            ->badge(),
                                                        TextEntry::make("module_extra_listing_price")
                                                            ->state($module->module_extra_listing_price)
                                                            ->label('Extra Listing Cost')
                                                            ->money('USD')
                                                            ->icon('heroicon-m-currency-dollar')
                                                            ->color('gray'),
                                                    ]),
                                            ])
                                    )->toArray()
                                ]),
                        ])->columnSpan(1),

                        // Right Column - User Features & Stats
                        Group::make([
                            Section::make('Member Features')
                                ->icon('heroicon-m-user-circle')
                                ->collapsible()
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            IconEntry::make('packageUserInfo.show_website_link')
                                                ->label('Website Link')
                                                ->boolean()
                                                ->color('success'),
                                            IconEntry::make('packageUserInfo.show_phone')
                                                ->label('Phone Number')
                                                ->boolean()
                                                ->color('success'),
                                            IconEntry::make('packageUserInfo.show_address')
                                                ->label('Address')
                                                ->boolean()
                                                ->color('success'),
                                            IconEntry::make('packageUserInfo.show_social_media_links')
                                                ->label('Social Media')
                                                ->boolean()
                                                ->color('success'),
                                            IconEntry::make('packageUserInfo.searchable')
                                                ->label('Searchable')
                                                ->boolean()
                                                ->color('success'),
                                            IconEntry::make('packageUserInfo.show_public_review')
                                                ->label('Reviews')
                                                ->boolean()
                                                ->color('success'),
                                            IconEntry::make('packageUserInfo.show_contact_form')
                                                ->label('Contact Form')
                                                ->boolean()
                                                ->color('success'),
                                        ]),
                                ]),

                            Section::make('Gallery Features')
                                ->icon('heroicon-m-photo')
                                ->collapsible()
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            Group::make([
                                                IconEntry::make('packageUserInfo.show_image_gallery')
                                                    ->label('Image Gallery')
                                                    ->boolean()
                                                    ->color('success'),
                                                TextEntry::make('packageUserInfo.image_gallery_limit')
                                                    ->label('Image Limit')
                                                    ->suffix(' images')
                                                    ->badge()
                                                    ->visible(fn ($record) => $record->packageUserInfo?->show_image_gallery),
                                            ])->columnSpan(1),

                                            Group::make([
                                                IconEntry::make('packageUserInfo.show_video_gallery')
                                                    ->label('Video Gallery')
                                                    ->boolean()
                                                    ->color('success'),
                                                TextEntry::make('packageUserInfo.video_gallery_limit')
                                                    ->label('Video Limit')
                                                    ->suffix(' videos')
                                                    ->badge()
                                                    ->visible(fn ($record) => $record->packageUserInfo?->show_video_gallery),
                                            ])->columnSpan(1),
                                        ]),
                                ]),

                            Section::make('Subscription Stats')
                                ->icon('heroicon-m-chart-bar')
                                ->collapsible()
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            TextEntry::make('monthly_subscribers')
                                                ->label('Monthly')
                                                ->state(fn ($record) => $record->subscriptions->where('payment_term', 'monthly')->where('is_active', true)->count())
                                                ->badge()
                                                ->color('info'),
                                            TextEntry::make('quarterly_subscribers')
                                                ->label('Quarterly')
                                                ->state(fn ($record) => $record->subscriptions->where('payment_term', 'quarterly')->where('is_active', true)->count())
                                                ->badge()
                                                ->color('info'),
                                            TextEntry::make('yearly_subscribers')
                                                ->label('Yearly')
                                                ->state(fn ($record) => $record->subscriptions->where('payment_term', 'yearly')->where('is_active', true)->count())
                                                ->badge()
                                                ->color('info'),
                                            TextEntry::make('auto_renewal')
                                                ->label('Auto-Renewal')
                                                ->state(fn ($record) => $record->subscriptions->where('auto_renewal', true)->where('is_active', true)->count())
                                                ->badge()
                                                ->color('success'),
                                        ]),
                                ]),
                        ])->columnSpan(1),
                    ])->columnSpanFull(),
            ]);
    }
}
