<?php

namespace App\Filament\Resources\Page;

use App\Filament\Resources\Page\PageResource\Pages;
use App\Filament\Resources\Page\PageResource\RelationManagers;
use App\Models\Page;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Illuminate\Filesystem\Filesystem;
use Symfony\Component\Finder\SplFileInfo;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Builder as FormBuilder;
use Filament\Forms\Components\Toggle;
use App\Filament\Traits\HasSeoSection;
use App\Enums\PageStatus;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Tables\Columns\TextColumn;
use FilamentTiptapEditor\TiptapEditor;
use FilamentTiptapEditor\Enums\TiptapOutput;
use Illuminate\Support\Facades\Route;
use App\Enums\PageTypeEnum;
use App\Models\Widget;
use Filament\Forms\Components\Repeater;

class PageResource extends Resource
{
    use HasSeoSection;

    protected static ?string $model = Page::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns(10)
                    ->schema([
                        Group::make()
                            ->columnSpan(7)
                            ->schema([
                                Section::make('Page Details')
                                    ->schema([
                                        TextInput::make('title')
                                            ->required()
                                            ->autofocus()
                                            ->maxLength(255)
                                            ->live(onBlur: true),
                                            // ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $operation) {
                                            //     dump($operation);
                                            //     if ($operation === 'update') {
                                            //         return;
                                            //     }
                                            //     if (($get('slug') ?? '') !== Str::slug($old)) {
                                            //         return;
                                            //     }
                                            //     $set('slug', Str::slug($state));
                                            // }),
                                        TextInput::make('heading')
                                            ->maxLength(255),
                                        TextInput::make('slug')
                                            ->required()
                                            ->reactive()
                                            ->dehydrated()
                                            ->unique(ignoreRecord: true),
                                        TiptapEditor::make('content')
                                            ->label('Content')
                                            ->placeholder('Enter the post content')
                                            ->profile('default')
                                            ->disk('media') // optional, defaults to config setting
                                            ->directory('blog') // optional, defaults to config setting
                                            ->maxSize(2024) // optional, defaults to config setting
                                            ->output(TiptapOutput::Json) // optional, change the format for saved data, default is html
                                            ->maxContentWidth('5xl')
                                            ->extraInputAttributes(['style' => 'max-height: 50rem; min-height: 20rem;']),
                                        ...static::getTemplateSchemas(),
                                        Repeater::make('pageWidgets')
                                            ->relationship()
                                            ->schema([
                                                Select::make('widget_id')
                                                    ->label('Widget')
                                                    ->relationship('widget', 'title')
                                                    ->searchable()
                                                    ->reactive()
                                                    ->preload()
                                                    ->required(),
                                                Toggle::make('enable')
                                                    ->label('Enable')
                                                    ->default(true),
                                            ])
                                            ->reorderableWithButtons()
                                            ->collapsible()
                                            ->addActionLabel('Add Widget')
                                            ->afterStateHydrated(function (Set $set, $state) {
                                                $set('pageWidgets', $state);
                                            })
                                            ->reorderable()
                                            ->collapsed()
                                            ->orderColumn('sort_order')
                                            ->columns(1)
                                            ->itemLabel(fn (array $state): ?string => 
                                                \App\Models\Widget::find($state['widget_id'])?->title ?? 'Widget'
                                            ),

                                    ]),
                                static::getSeoSection(),
                            ]),
                        Group::make()
                            ->columnSpan(3)
                            ->schema([
                                Section::make('Template')
                                    ->schema([
                                        Select::make('status')
                                            ->options(PageStatus::class)
                                            ->default(PageStatus::Published)
                                            ->required()
                                            ->native(false),
                                        Select::make('page_type')
                                            ->options(PageTypeEnum::class)
                                            ->default(PageTypeEnum::Page->value)
                                            ->reactive()
                                            ->native(false)
                                            ->required(),
                                        Select::make('route_name')
                                            ->options(static::getRouteNameOptions())
                                            ->searchable()
                                            ->native(false)
                                            ->reactive()
                                            ->unique(ignoreRecord: true)
                                            ->visible(function (Get $get) {
                                                return $get('page_type') == PageTypeEnum::Route->value;
                                            })
                                            ->requiredIf('page_type', PageTypeEnum::Route->value),
                                        Select::make('template')
                                            ->reactive()
                                            ->options(static::getTemplates())
                                            ->requiredIf('page_type', PageTypeEnum::Page->value)
                                            ->native(false)
                                            ->visible(function (Get $get) {
                                                return $get('page_type') == PageTypeEnum::Page->value;
                                            }),
                                        Select::make('parent_id')
                                            ->options(Page::all()->pluck('title', 'id'))
                                            ->native(false)
                                            ->reactive()
                                            ->visible(function (Get $get) {
                                                return $get('page_type') == PageTypeEnum::Page->value;
                                            }),
                                        SpatieMediaLibraryFileUpload::make('featured_image')
                                            ->collection('featured_image')
                                            ->imageEditor()
                                            ->hintIcon('heroicon-m-photo')
                                            ->conversion('thumb'),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('title'),
                TextColumn::make('slug'),
                TextColumn::make('status')
                    ->badge(),
                TextColumn::make('template'),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPages::route('/'),
            'create' => Pages\CreatePage::route('/create'),
            'edit' => Pages\EditPage::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }

    public static function getTemplates(): Collection
    {
        return static::getTemplateClasses()->mapWithKeys(fn($class) => [$class => $class::title()]);
    }

    public static function getTemplateClasses(): Collection
    {
        $filesystem = app(Filesystem::class);

        return collect($filesystem->allFiles(app_path('Filament/PageTemplates')))
            ->map(function (SplFileInfo $file): string {
                return (string) Str::of('App\\Filament\\PageTemplates')
                    ->append('\\', $file->getRelativePathname())
                    ->replace(['/', '.php'], ['\\', '']);
            })
            ->filter(fn(string $class): bool => class_exists($class) && ! (new \ReflectionClass($class))->isAbstract() && $class !== \App\Filament\PageTemplates\Widgets\Widget::class);
    }

    public static function getTemplateSchemas(): array
    {
        return static::getTemplateClasses()
            ->map(
                fn($class) =>
                Forms\Components\Group::make($class::schema())
                    ->columnSpan(1)
                    ->afterStateHydrated(fn($component, $state) => $component->getChildComponentContainer()->fill($state))
                    ->statePath('data.' . static::getTemplateName($class))
                    ->visible(fn($get) => $get('template') === $class)
            )
            ->toArray();
    }

    public static function getTemplateName($class)
    {
        return Str::of($class)->afterLast('\\')->snake()->toString();
    }

    public static function getRouteNameOptions(): Collection
    {
        $excludedRoutes = config('filament-menu-builder.exclude_route_names', []);
        $routes = collect(Route::getRoutes())
            ->filter(function ($route) use ($excludedRoutes) {
                $routeName = $route->getName();
                if (! $routeName) {
                    return false;
                }

                // Check if the route name matches any of the excluded patterns
                $isExcluded = false;
                foreach ($excludedRoutes as $pattern) {
                    if (preg_match($pattern, $routeName)) {
                        $isExcluded = true;

                        break;
                    }
                }

                return ! $isExcluded;
            })
            ->mapWithKeys(function ($route) {
                return [$route->getName() => $route->getName()];
            });

        return $routes;
    }
}
