<?php

namespace App\Filament\Resources\Page\PageResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Get;
use Filament\Forms\Set;
use App\Filament\PageTemplates\Widgets\Widget as BaseWidget;
use Illuminate\Support\Collection;
use Illuminate\Filesystem\Filesystem;
use Symfony\Component\Finder\SplFileInfo;
use Illuminate\Support\Str;
use App\Enums\WidgetComponentType;

class WidgetsRelationManager extends RelationManager
{
    protected static string $relationship = 'widgets';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Group::make()
                    ->schema([
                        TextInput::make('title')
                            ->required()
                            ->maxLength(255),
                        Select::make('component')
                            ->options(static::getWidgetClasses())
                            ->label('Component')
                            ->reactive()
                            ->afterStateUpdated(function (Set $set, ?string $state) {
                                if ($state) {
                                    $set('component', $state::$component);
                                    $set('component_type', $state::$componentType);
                                }
                            })
                            ->required(),
                        Select::make('component')
                            ->dehydrated()
                            ->options(static::getWidgetClasses())
                            ->required(),
                        Select::make('component_type')
                            ->label('Component Type')
                            ->options(WidgetComponentType::class)
                            ->dehydrated()
                            ->required(),
                        Forms\Components\Group::make(static::getWidgetSchemas())
                            ->columnSpan(1)
                            ->afterStateHydrated(fn($component, $state) => $component->getChildComponentContainer()->fill($state))
                            ->statePath('data')
                            ->visible(fn(Get $get) => $get('widget_class') !== null),
                    ])->columnSpan(2),
                Forms\Components\Group::make()
                    ->schema([
                        TextInput::make('sort_order')
                            ->numeric()
                            ->default(0),
                        Toggle::make('enable')
                            ->default(true),
                    ])->columnSpan(1),
            ])->columns(3);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('title')
            ->columns([
                Tables\Columns\TextColumn::make('title'),
                Tables\Columns\TextColumn::make('component'),
                Tables\Columns\TextColumn::make('component_type'),
                Tables\Columns\TextColumn::make('pivot.sort_order'),
                Tables\Columns\IconColumn::make('pivot.enable')
                    ->boolean(),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\AttachAction::make()
                    ->form(fn (Tables\Actions\AttachAction $action): array => [
                        Forms\Components\Group::make(static::form(new Form($action))->getSchema()),
                        Forms\Components\Group::make()
                            ->schema([
                                TextInput::make('sort_order')
                                    ->numeric()
                                    ->default(0),
                                Toggle::make('enable')
                                    ->default(true),
                            ])->columnSpan(1),
                    ]),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DetachAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DetachBulkAction::make(),
                ]),
            ]);
    }

    public static function getWidgetClasses(): Collection
    {
        $filesystem = app(Filesystem::class);

        return collect($filesystem->allFiles(app_path('Filament/PageTemplates/Widgets')))
            ->map(function (SplFileInfo $file): string {
                return (string) Str::of('App\Filament\PageTemplates\Widgets')
                    ->append('\\', $file->getRelativePathname())
                    ->replace(['/', '.php'], ['\\', '']);
            })
            ->filter(fn(string $class): bool => class_exists($class) && ! (new \ReflectionClass($class))->isAbstract() && $class !== BaseWidget::class)
            ->mapWithKeys(fn($class) => [$class => $class::title()]);
    }

    public static function getWidgetSchemas(): array
    {
        return static::getWidgetClasses()
            ->map(
                fn($class) =>
                Forms\Components\Group::make($class::schema())
                    ->columnSpan(1)
                    ->afterStateHydrated(fn($component, $state) => $component->getChildComponentContainer()->fill($state))
                    ->statePath('data')
                    ->visible(fn(Get $get) => $get('widget_class') === $class)
            )
            ->toArray();
    }
}
