<?php

namespace App\Filament\Resources\Page;

use App\Filament\Resources\Page\WidgetResource\Pages;
use App\Enums\WidgetComponentType;
use App\Models\Widget;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Illuminate\Filesystem\Filesystem;
use Symfony\Component\Finder\SplFileInfo;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Forms\Components\Hidden;
use Filament\Tables\Columns\TextColumn;


class WidgetResource extends Resource 
{
    

    protected static ?string $model = Widget::class;


    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $navigationGroup = 'Pages';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Widget Details')
                    ->schema([
                        Grid::make()
                            ->columns(2)
                            ->schema([
                                
                                TextInput::make('title')
                                    ->required()
                                    ->maxLength(255),
                                Select::make('component')
                                    ->reactive()
                                    ->options(static::getWidgetTemplates())
                                    ->afterStateUpdated(function (Set $set, ?string $state) {
                                        if (! $state) {
                                            return;
                                        }
                                        $set('component_type', static::resolveComponentTypeFromComponent($state));
                                    }),
                                Hidden::make('component_type')
                                    ->dehydrated(),
                            ]),

                        ...static::getWidgetSchemas(),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('title'),
                TextColumn::make('component'),
                TextColumn::make('component_type'),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListWidgets::route('/'),
            'create' => Pages\CreateWidget::route('/create'),
            'edit' => Pages\EditWidget::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }

    public static function getWidgetTemplates(): Collection
    {
        return static::getWidgetClasses()->mapWithKeys(fn($class) => [$class::getComponent() => $class::title()]);
    }

    public static function getWidgetClasses(): Collection
    {
        $filesystem = app(Filesystem::class);

        return collect($filesystem->allFiles(app_path('Filament/PageTemplates/Widgets')))
            ->map(function (SplFileInfo $file): string {
                return (string) Str::of('App\\Filament\\PageTemplates\\Widgets')
                    ->append('\\', $file->getRelativePathname())
                    ->replace(['/', '.php'], ['\\', '']);
            })
            ->filter(fn(string $class): bool => class_exists($class) && ! (new \ReflectionClass($class))->isAbstract() && $class !== \App\Filament\PageTemplates\Widgets\Widget::class);
    }

    public static function getWidgetSchemas(): array
    {
        return static::getWidgetClasses()
            ->map(
                fn($class) =>
                Forms\Components\Group::make($class::schema())
                    ->columnSpan(1)
                    ->afterStateHydrated(fn($component, $state) => $component->getChildComponentContainer()->fill($state))
                    ->statePath('data.' . self::getWidgetName($class))
                    ->visible(fn(Get $get) => $get('component') === $class::getComponent())
            )
            ->toArray();
    }

    public static function getWidgetName($class)
    {
        return Str::of($class)->afterLast('\\')->snake()->toString();
    }

    public static function resolveComponentTypeFromComponent(string $component): string
    {
        foreach (static::getWidgetClasses() as $class) {
            if ($class::getComponent() === $component) {
                return $class::getComponentType();
            }
        }

        return WidgetComponentType::Livewire->value;
    }

}
