<?php

namespace App\Filament\Resources\Product;

use App\Filament\Resources\Product\ProductCategoryResource\Pages;
use App\Filament\Resources\Product\ProductCategoryResource\RelationManagers;
use App\Models\Product\ProductCategory;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Illuminate\Support\Str;
use Filament\Forms\Get;
use Filament\Forms\Set;

class ProductCategoryResource extends Resource
{
    protected static ?string $model = ProductCategory::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 10, // 10-column grid for 70/30 split
                    ])
                    ->schema([
                        // Main Content Column (7/10 = 70%)
                        Group::make()
                            ->columnSpan(['lg' => 7])
                            ->schema([
                                Section::make('Category Details')
                                    ->schema([
                                        TextInput::make('product_category_name')
                                            ->required()
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                                if ($context === 'create' && empty($get('slug'))) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            }),
                                        TextInput::make('slug')
                                            ->required()
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (?string $state, Set $set) {
                                                if ($state) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            }),
                                        Textarea::make('product_category_description')
                                            ->columnSpanFull()
                                            ->rows(4),
                                    ]),
                            ]),

                        // Sidebar Column (3/10 = 30%)
                        Group::make()
                            ->columnSpan(['lg' => 3])
                            ->schema([
                                Section::make('Settings')
                                    ->schema([
                                        Select::make('parent_id')
                                            ->relationship(
                                                name: 'parent',
                                                titleAttribute: 'product_category_name',
                                                modifyQueryUsing: fn(Builder $query) => $query
                                                    ->select(['product_categories.id', 'product_categories.product_category_name'])
                                            )
                                            ->searchable()
                                            ->preload()
                                            ->placeholder('Select parent category'),
                                        Toggle::make('is_active')
                                            ->label('Active Status')
                                            ->helperText('Toggle to enable/disable this category')
                                            ->default(true)
                                            ->required()
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                
                Tables\Columns\TextColumn::make('product_category_name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('slug')
                    ->searchable(),
                Tables\Columns\TextColumn::make('parent.product_category_name')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\IconColumn::make('is_active')
                    ->boolean(),
                Tables\Columns\TextColumn::make('created_by')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProductCategories::route('/'),
            'create' => Pages\CreateProductCategory::route('/create'),
            'view' => Pages\ViewProductCategory::route('/{record}'),
            'edit' => Pages\EditProductCategory::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
