<?php

namespace App\Filament\Resources\Product;

use App\Filament\Resources\Product\ProductResource\Pages;
use App\Filament\Resources\Product\ProductResource\RelationManagers;
use App\Models\Product\Product;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\DateTimePicker;
use Illuminate\Support\Str;
use Filament\Forms\Get;
use Filament\Forms\Set;
use App\Filament\Traits\HasSeoSection;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;

class ProductResource extends Resource
{
    use HasSeoSection;

    protected static ?string $model = Product::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 10, // 10-column grid for 70/30 split
                    ])
                    ->schema([
                        // Main Content Column (7/10 = 70%)
                        Group::make()
                            ->columnSpan(['lg' => 7])
                            ->schema([
                                Section::make('Product Details')
                                    ->schema([
                                        TextInput::make('product_title')
                                            ->required()
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state, $context) {
                                                if ($context === 'create' && empty($get('slug'))) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            }),
                                        TextInput::make('slug')
                                            ->required()
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (?string $state, Set $set) {
                                                if ($state) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            }),
                                        TextInput::make('product_sku')
                                            ->maxLength(255),
                                        Textarea::make('product_description')
                                            ->columnSpanFull(),
                                        Textarea::make('link')
                                            ->columnSpanFull(),
                                    ]),

                                // Gallery Images Section
                                Section::make('Product Images')
                                    ->schema([
                                        SpatieMediaLibraryFileUpload::make('gallery_images')
                                            ->collection('gallery_images')
                                            ->multiple()
                                            ->reorderable()
                                            ->maxFiles(10)
                                            ->image()
                                            ->imageEditor()
                                            ->columnSpanFull()
                                            ->hint('Upload up to 10 images. Recommended size: 1200x800px')
                                            ->imageResizeMode('cover')
                                            ->imageCropAspectRatio('3:2')
                                            ->imageResizeTargetWidth('1200')
                                            ->imageResizeTargetHeight('800'),
                                    ]),

                                // Add SEO Section
                                static::getSeoSection(),
                            ]),

                        // Sidebar Column (3/10 = 30%)
                        Group::make()
                            ->columnSpan(['lg' => 3])
                            ->schema([
                                Section::make('Settings')
                                    ->schema([
                                        Select::make('parent_id')
                                            ->relationship('parent', 'product_title')
                                            ->searchable()
                                            ->preload()
                                            ->placeholder('Select parent product'),
                                        TextInput::make('quantity')
                                            ->required()
                                            ->numeric()
                                            ->default(0),
                                        TextInput::make('price')
                                            ->required()
                                            ->numeric()
                                            ->default(0.00)
                                            ->prefix('$'),
                                        DateTimePicker::make('published_at')
                                            ->label('Published Date'),
                                        DateTimePicker::make('expired_at')
                                            ->label('Expiry Date'),
                                        Select::make('categories')
                                            ->multiple()
                                            ->relationship(
                                                name: 'categories',
                                                titleAttribute: 'product_category_name',
                                                modifyQueryUsing: fn(Builder $query) => $query
                                                    ->select(['product_categories.id', 'product_categories.product_category_name'])
                                            )
                                            ->createOptionForm([
                                                TextInput::make('product_category_name')
                                                    ->required()
                                                    ->maxLength(255)
                                                    ->live(onBlur: true)
                                                    ->afterStateUpdated(function (string $state, Set $set) {
                                                        $set('slug', Str::slug($state));
                                                    }),
                                                TextInput::make('slug')
                                                    ->required()
                                                    ->maxLength(255)
                                                    ->unique('product_categories', 'slug'),
                                                TextInput::make('created_by')
                                                    ->default(auth()->guard('web')->user()?->id)
                                                    ->hidden(),
                                            ])
                                            ->createOptionAction(
                                                fn(Forms\Components\Actions\Action $action) => $action
                                                    ->modalHeading('Create Category')
                                                    ->modalWidth('sm')
                                            )
                                            ->preload()
                                            ->searchable()
                                            ->optionsLimit(100)
                                            ->loadingMessage('Loading categories...')
                                            ->noSearchResultsMessage('No categories found.')
                                            ->placeholder('Select categories')
                                            ->helperText('Choose one or more categories'),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('product_title')
                    ->searchable(),
                Tables\Columns\TextColumn::make('slug')
                    ->searchable(),
                Tables\Columns\TextColumn::make('product_sku')
                    ->searchable(),
                Tables\Columns\TextColumn::make('quantity')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('price')
                    ->money()
                    ->sortable(),
                Tables\Columns\TextColumn::make('published_at')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('expired_at')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProducts::route('/'),
            'create' => Pages\CreateProduct::route('/create'),
            'view' => Pages\ViewProduct::route('/{record}'),
            'edit' => Pages\EditProduct::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
