<?php

namespace App\Filament\Resources\Project;

use App\Filament\Resources\Project\ProjectResource\Traits\HasProjectForm;
use Filament\Forms;
use Filament\Tables;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Illuminate\Support\Str;
use App\Enums\ProjectStatus;
use App\Models\Project\Project;
use Filament\Resources\Resource;
use App\Models\Project\ProjectType;
use Filament\Forms\Components\Grid;
use Filament\Tables\Actions\Action;
use Filament\Forms\Components\Group;
use Filament\Support\Enums\MaxWidth;
use App\Enums\ProjectCompletedStatus;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Textarea;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\ViewAction;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\ActionGroup;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\RichEditor;
use Filament\Tables\Actions\DeleteAction;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Forms\Components\DateTimePicker;
use Filament\Tables\Actions\DeleteBulkAction;
use Cheesegrits\FilamentGoogleMaps\Fields\Map;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Resources\Project\ProjectResource\Pages;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Forms\Components\Actions\Action as FormAction;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Illuminate\Support\Facades\Auth;

class ProjectResource extends Resource
{
    use HasProjectForm;
    protected static ?string $model = Project::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form->schema((new static)->getProjectFormSchema(panel: 'admin'));
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->tooltip(fn($state) => $state)
                    ->formatStateUsing(fn($state) => Str::limit($state, 20)),
                Tables\Columns\TextColumn::make('types.type_name')
                    ->label('Type')
                    ->wrap(),
                Tables\Columns\TextColumn::make('project_status')
                    ->label('Status')
                    ->badge()
                    ->searchable(),
                Tables\Columns\TextColumn::make('user.name')
                    ->label('Belongs To')
                    ->sortable(),
                Tables\Columns\TextColumn::make('published_at')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('expired_at')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\SelectFilter::make('team_member_status')
                    ->options([
                        ProjectStatus::Draft->value => ProjectStatus::Draft->name,
                        ProjectStatus::Published->value => ProjectStatus::Published->name,
                        ProjectStatus::Pending->value => ProjectStatus::Pending->name,
                        ProjectStatus::Expired->value => ProjectStatus::Expired->name,
                        ProjectStatus::Trashed->value => ProjectStatus::Trashed->name,
                    ])
                    ->label('Status'),
                Tables\Filters\SelectFilter::make('user_id')
                    ->relationship('user', 'name')
                    ->label('Belongs To')
                    ->preload()
                    ->searchable(),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->actions([
                ActionGroup::make([

                    Action::make('approve')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(
                            fn(Project $record): bool =>
                            $record->project_status->value === ProjectStatus::Pending->value
                        )
                        ->action(function (Project $record): void {
                            $record->project_status = ProjectStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Project')
                        ->modalDescription('Are you sure you want to approve this project? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, approve')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Project Approved')
                                ->body('Project has been approved successfully')
                        ),
                    Action::make('publish')
                        ->icon('heroicon-o-arrow-up-circle')
                        ->color('success')
                        ->visible(
                            fn(Project $record): bool =>
                            in_array($record->project_status->value, [ProjectStatus::Draft->value, ProjectStatus::Expired->value])
                        )
                        ->action(function (Project $record): void {
                            $record->project_status = ProjectStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Publish Project')
                        ->modalDescription('Are you sure you want to publish this project? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, Publish')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Project Published')
                                ->body('Project has been published successfully')
                        ),
                    ViewAction::make()
                        ->slideOver()
                        ->hidden(fn(Project $record): bool => $record->trashed()),
                    EditAction::make()
                        ->hidden(fn(Project $record): bool => $record->trashed()),
                    ReplicateAction::make()
                        ->before(function (Project $record): void {
                            $record->project_status = ProjectStatus::Draft->value;
                        })
                        ->after(function (Project $replica, Project $record): void {
                            $replica->types()->sync($record->types->pluck('id'));
                        })
                        ->hidden(fn(Project $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(Project $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProjects::route('/'),
            'create' => Pages\CreateProject::route('/create'),
            'edit' => Pages\EditProject::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
