<?php

namespace App\Filament\Resources\Project\ProjectResource\Pages;

use App\Filament\Resources\Project\ProjectResource;
use Filament\Actions;
use Filament\Resources\Pages\CreateRecord;
use App\Enums\ProjectStatus;
use Filament\Notifications\Notification;
use Illuminate\View\View;

class CreateProject extends CreateRecord
{
    protected static string $resource = ProjectResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('publish')
                ->label(fn () => match($this->data['project_status'] ?? null) {
                    ProjectStatus::Pending->value => 'Save',
                    default => 'Publish',
                })
                ->icon(fn () => match($this->data['project_status'] ?? null) {
                    ProjectStatus::Pending->value => 'heroicon-o-document-text',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn () => match($this->data['project_status'] ?? null) {
                    ProjectStatus::Pending->value => 'primary',
                    default => 'success',
                })
                ->action(function () {
                    $data = $this->form->getState();

                    if ($data['project_status'] === ProjectStatus::Draft->value || $data['project_status'] === ProjectStatus::Published->value) {
                        $this->data['project_status'] = ProjectStatus::Published->value;
                        $this->data['published_at'] = now();
                    }
                    
                    $this->create();
                })
                ->keyBindings(['mod+s']),

            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['project_status'] = ProjectStatus::Draft->value;
                    $this->create();
                }),

            Actions\Action::make('cancel')
                ->label('Cancel')
                ->icon('heroicon-o-x-mark')
                ->color('gray')
                ->url(fn () => ProjectResource::getUrl()),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            Actions\Action::make('create')
                ->label(fn () => match($this->data['project_status'] ?? null) {
                    ProjectStatus::Pending->value => 'Save',
                    default => 'Publish',
                })
                ->icon(fn () => match($this->data['project_status'] ?? null) {
                    ProjectStatus::Pending->value => 'heroicon-o-document-text',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn () => match($this->data['project_status'] ?? null) {
                    ProjectStatus::Pending->value => 'primary',
                    default => 'success',
                })
                ->action(function () {
                    if ($this->data['project_status'] === ProjectStatus::Draft->value || $this->data['project_status'] === ProjectStatus::Published->value) {
                        $this->data['project_status'] = ProjectStatus::Published->value;
                        $this->data['published_at'] = now();
                    }
                    
                    $this->create();
                })
                ->keyBindings(['mod+s']),
            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['project_status'] = ProjectStatus::Draft->value;
                    $this->create();
                }),
            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        // If the post status is set to Published, set the published_at timestamp to now
        if ($data['project_status'] === ProjectStatus::Published->value) {
            $data['published_at'] = now();
        }
        
        return $data;
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('edit', ['record' => $this->record]);
    }

    protected function getCreatedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match($this->data['project_status']) {
                ProjectStatus::Draft->value => 'Draft Saved',
                ProjectStatus::Pending->value => 'Project Saved',
                default => 'Project Published',
            })
            ->body(match($this->data['project_status']) {
                ProjectStatus::Draft->value => 'Your project has been saved as draft.',
                ProjectStatus::Pending->value => 'Your project has been saved as pending.',
                default => 'Your project has been published successfully.',
            });
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
