<?php

namespace App\Filament\Resources\Project\ProjectResource\Pages;

use App\Filament\Resources\Project\ProjectResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use Filament\Notifications\Notification;
use Illuminate\View\View;
use App\Enums\ProjectStatus;
use App\Models\Location\Country;

class EditProject extends EditRecord
{
    protected static string $resource = ProjectResource::class;

    protected function getSavedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match ($this->data['project_status']) {
                ProjectStatus::Draft->value => 'Draft Updated',
                ProjectStatus::Pending->value => 'Project Updated',
                default => 'Project Updated',
            })
            ->body(match ($this->data['project_status']) {
                ProjectStatus::Draft->value => 'Your project has been updated and saved as draft.',
                ProjectStatus::Pending->value => 'Your project has been updated and saved as pending.',
                default => 'Your project has been updated successfully.',
            });
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('save')
                ->label('Update')
                ->action('save')
                ->icon('heroicon-o-check-circle')
                ->color('primary')
                ->keyBindings(['mod+s']),



            Actions\ActionGroup::make([
                Actions\ActionGroup::make([
                    // Array of action

                    Actions\DeleteAction::make()
                        ->label('Move to Trash')
                        ->icon('heroicon-o-trash')
                        ->requiresConfirmation()
                        ->hidden(fn(): bool => $this->record->trashed())
                        ->modalHeading('Delete Project')
                        ->modalDescription('Are you sure you want to delete this project? This action can be undone later.')
                        ->modalSubmitActionLabel('Yes, delete it')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Project deleted')
                                ->body('The project has been moved to trash.')
                        ),
                ])->dropdown(false),
                Actions\Action::make('cancel')
                    ->label('Cancel')
                    ->icon('heroicon-o-x-mark')
                    ->color('gray')
                    ->url(fn() => ProjectResource::getUrl()),
            ])
                ->tooltip('More Actions')
                ->icon('heroicon-m-ellipsis-vertical')
                ->color('info')
                ->button()
                ->extraAttributes([
                    'class' => 'more-actions-btn',
                ]),
        ];
    }

    protected function afterDelete(): void
    {
        if (! $this->record->trashed()) {
            $this->record->update(['project_status' => ProjectStatus::Trashed]);
        }
    }

    protected function afterRestore(): void
    {
        $this->record->update(['project_status' => ProjectStatus::Draft]);
    }

    public function afterSave(): void
    {
        if ($this->record->project_status !== ProjectStatus::Trashed && $this->record->trashed()) {
            $this->record->restore();
        }

        $country = Country::firstOrCreate(
            ['country_code' => $this->data['country_code']],
            [
                'country_name' => $this->data['country_name'],
                'country_code' => $this->data['country_code'],
                'created_by' => auth()->id()
            ]
        );

        if ($this->data['state_name']) {
            $state = \App\Models\Location\State::firstOrCreate(
                [
                    'country_id' => $country->id,
                    'state_name' => $this->data['state_name']
                ],
                [
                    'created_by' => auth()->id()
                ]
            );

            $this->record->state_id = $state->id;

            if ($this->data['city_name']) {
                $city = \App\Models\Location\City::firstOrCreate(
                    [
                        'country_id' => $country->id,
                        'state_id' => $state->id,
                        'city_name' => $this->data['city_name']
                    ],
                    [
                        'created_by' => auth()->id()
                    ]
                );

                $this->record->city_id = $city->id;
            }
        }

        if ($this->data['country_name']) {
            $this->record->country_id = $country->id;
        }

        $this->record->save();
    }

    protected function mutateFormDataBeforeFill(array $data): array
    {
        if (isset($data['country_id'])) {
            $country = Country::find($data['country_id']);
            if ($country) {
                $data['country_name'] = $country->country_name;
                $data['country_code'] = $country->country_code;
            }
        }

        if (isset($data['state_id'])) {
            $state = \App\Models\Location\State::find($data['state_id']);
            if ($state) {
                $data['state_name'] = $state->state_name;
            }
        }

        if (isset($data['city_id'])) {
            $city = \App\Models\Location\City::find($data['city_id']);
            if ($city) {
                $data['city_name'] = $city->city_name;
            }
        }

        return $data;
    }

    protected function getFormActions(): array
    {
        return [
            $this->getSaveFormAction()
                ->label('Update')
                ->icon('heroicon-o-check-circle'),

            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
