<?php

namespace App\Filament\Resources\Project\ProjectResource\Traits;

use App\Enums\ProjectCompletedStatus;
use App\Enums\ProjectStatus;
use Cheesegrits\FilamentGoogleMaps\Fields\Map;
use Filament\Forms\Components\Actions\Action as FormAction;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Livewire\Component as Livewire;

trait HasProjectForm
{
    public function getProjectFormSchema($context = 'create', $panel = 'dashboard'): array
    {
        return [
            Grid::make()
                ->columns([
                    'default' => 1,
                    'lg' => 10,
                ])
                ->schema([
                    // Main Content Column (70%)
                    Group::make()
                        ->columnSpan(['lg' => 7])
                        ->schema([
                            Section::make('Basic Information')
                                ->description('Enter the core details of the project')
                                ->schema([
                                    TextInput::make('title')
                                        ->required()
                                        ->maxLength(191)
                                        ->placeholder('Enter the title for the project')
                                        ->live(onBlur: true)
                                        ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state) use ($context) {
                                            if ($context === 'create' && empty($get('slug'))) {
                                                $set('slug', Str::slug($state));
                                            }
                                        })
                                        ->validationMessages([
                                            'regex' => 'Title must only contain letters, numbers, spaces, and &.',
                                        ]),
                                    TextInput::make('slug')
                                        ->required()
                                        ->placeholder('Enter the slug for the project')
                                        ->helperText('Will be automatically generated from title if left empty.')
                                        ->maxLength(255)
                                        ->live(onBlur: true)
                                        ->afterStateUpdated(function (?string $state, Set $set) {
                                            if ($state) {
                                                $set('slug', Str::slug($state));
                                            }
                                        })
                                        ->dehydrated()
                                        ->unique(ignoreRecord: true),
                                    RichEditor::make('description')
                                        ->label('Project Overview')
                                        ->disableToolbarButtons([
                                            'codeBlock'
                                        ])
                                        ->placeholder('Enter the description for the project')
                                        ->columnSpanFull(),
                                ]),
                            Section::make('Gallery')
                                ->schema([
                                    SpatieMediaLibraryFileUpload::make('gallery')
                                        ->collection('gallery')
                                        ->helperText('Add images to the gallery for the project.')
                                        ->multiple()
                                        ->reorderable()
                                        ->image()
                                        ->preserveFilenames()
                                        ->maxSize(5120)
                                        ->label('Gallery')
                                        ->extraAttributes([
                                            'class' => 'upload-media-gallery',
                                        ]),
                                ]),
                            Section::make('Project Details')
                                ->schema([
                                    Textarea::make('techniques')
                                        ->placeholder('Enter the techniques for the project')
                                        ->rows(3),
                                    Textarea::make('equipments')
                                        ->label('Equipment')
                                        ->placeholder('Enter the equipments for the project')
                                        ->rows(3),
                                ]),
                            Section::make('Project Location')
                                ->schema([
                                    Grid::make()
                                        ->columns(2)
                                        ->schema([
                                            TextInput::make('location')
                                                ->columnSpanFull()
                                                ->required(),
                                            // First Column
                                            Group::make()
                                                ->columnSpan(1)
                                                ->schema([
                                                    Map::make('map_address')
                                                        ->mapControls([
                                                            'mapTypeControl'    => true,
                                                            'scaleControl'      => true,
                                                            'streetViewControl' => true,
                                                            'rotateControl'     => true,
                                                            'fullscreenControl' => true,
                                                            'searchBoxControl'  => false,
                                                            'zoomControl'       => true,
                                                        ])
                                                        ->defaultZoom(13)
                                                        ->defaultLocation([0, 0])
                                                        ->draggable()
                                                        ->clickable()
                                                        ->debug()
                                                        ->autocomplete('location')
                                                        ->autocompleteReverse()
                                                        ->reverseGeocode([
                                                            'address' => '%n %S %D %L %z',
                                                            'postal_code' => '%z',
                                                            'city_name' => '%L',
                                                            'state_name' => '%A1',
                                                            'state_code' => '%a1',
                                                            'country_name' => '%C',
                                                            'country_code' => '%c',
                                                        ])
                                                        ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                                            if (isset($state['lat']) && isset($state['lng'])) {
                                                                $set('latitude', $state['lat']);
                                                                $set('longitude', $state['lng']);

                                                                if (isset($state['address'])) {
                                                                    $set('address', $state['address']);
                                                                }
                                                                if (isset($state['postal_code'])) {
                                                                    $set('postal_code', $state['postal_code']);
                                                                }
                                                                if (isset($state['country_name'])) {
                                                                    $set('country_name', $state['country_name']);
                                                                }
                                                                if (isset($state['country_code'])) {
                                                                    $set('country_code', $state['country_code']);
                                                                }
                                                                if (isset($state['state_name'])) {
                                                                    $set('state_name', $state['state_name']);
                                                                }
                                                                if (isset($state['state_code'])) {
                                                                    $set('state_code', $state['state_code']);
                                                                }
                                                                if (isset($state['city_name'])) {
                                                                    $set('city_name', $state['city_name']);
                                                                }
                                                            }
                                                        })
                                                        ->afterStateHydrated(function ($component, $state, $record) {
                                                            if ($record && $record->latitude && $record->longitude) {
                                                                $component->state([
                                                                    'lat' => (float)$record->latitude,
                                                                    'lng' => (float)$record->longitude,
                                                                    'address' => $record->map_address ?: $record->address
                                                                ]);
                                                            }
                                                        }),
                                                ]),
                                            // Second Column
                                            Group::make()
                                                ->columnSpan(1)
                                                ->schema([
                                                    Grid::make()
                                                        ->columns(2)
                                                        ->schema([
                                                            TextInput::make('address')
                                                                ->label('Street Address')
                                                                ->required()
                                                                ->columnSpanFull(),
                                                            TextInput::make('latitude')
                                                                ->numeric()
                                                                ->columnSpan(1),
                                                            TextInput::make('longitude')
                                                                ->numeric()
                                                                ->columnSpan(1),
                                                            TextInput::make('country_name')
                                                                ->label('Country')
                                                                ->columnSpan(1)
                                                                ->readOnly(),
                                                            TextInput::make('state_name')
                                                                ->label('State')
                                                                ->columnSpan(1)
                                                                ->readOnly(),
                                                            TextInput::make('city_name')
                                                                ->label('City')
                                                                ->columnSpan(1)
                                                                ->readOnly(),
                                                            TextInput::make('postal_code')
                                                                ->label('Postal Code'),
                                                            Hidden::make('state_code'),
                                                            Hidden::make('country_code'),
                                                            Hidden::make('country_id'),
                                                            Hidden::make('state_id'),
                                                            Hidden::make('city_id'),
                                                        ]),
                                                ]),
                                        ]),
                                ]),
                            Section::make('Project Content')
                                ->schema([
                                    RichEditor::make('introduction')
                                        ->placeholder('Enter the introduction for the project')
                                        ->toolbarButtons(['bold', 'italic', 'bulletList', 'orderedList',])
                                        ->columnSpanFull(),
                                    RichEditor::make('objective')
                                        ->placeholder('Enter the objective for the project')
                                        ->toolbarButtons(['bold', 'italic', 'bulletList', 'orderedList',])
                                        ->columnSpanFull(),
                                    RichEditor::make('challenges')
                                        ->placeholder('Enter the challenges for the project')
                                        ->toolbarButtons(['bold', 'italic', 'bulletList', 'orderedList',])
                                        ->columnSpanFull(),
                                    RichEditor::make('solutions')
                                        ->placeholder('Enter the solutions for the project')
                                        ->toolbarButtons(['bold', 'italic', 'bulletList', 'orderedList',])
                                        ->columnSpanFull(),
                                    RichEditor::make('outcomes')
                                        ->placeholder('Enter the outcomes for the project')
                                        ->toolbarButtons(['bold', 'italic', 'bulletList', 'orderedList',])
                                        ->columnSpanFull(),
                                    Repeater::make('metrics')
                                        ->schema([
                                            TextInput::make('key')
                                                ->required()
                                                ->label('Metric Name')
                                                ->placeholder('Metric Name (e.g. Total Depth)')
                                                ->maxLength(20),
                                            TextInput::make('value')
                                                ->required()
                                                ->label('Metric Value')
                                                ->placeholder('Metric Value (e.g. 3,200 ft)')
                                                ->maxLength(20),
                                        ])
                                        ->columns(2)
                                        ->addActionLabel('Add Metric')
                                        ->reorderable()
                                        ->collapsible()
                                        ->itemLabel(fn(array $state): ?string => $state['key'] ?? null)
                                        ->columnSpanFull(),
                                ]),
                        ]),
                    // Sidebar Column (30%)
                    Group::make()
                        ->columnSpan(['lg' => 3])
                        ->schema([
                            Section::make('Project Settings')
                                ->schema(array_filter([
                                    // Admin-only fields
                                    $panel === 'admin' ? Select::make('project_status')
                                        ->options([
                                            ProjectStatus::Draft->value => ProjectStatus::Draft->name,
                                            ProjectStatus::Published->value => ProjectStatus::Published->name,
                                            ProjectStatus::Pending->value => ProjectStatus::Pending->name,
                                            ProjectStatus::Expired->value => ProjectStatus::Expired->name,
                                        ])
                                        ->helperText('Set the current status of your project')
                                        ->required()
                                        ->live()
                                        ->default(ProjectStatus::Draft->value) : null,
                                    $panel === 'admin' ? Select::make('user_id')
                                        ->label('Belongs To')
                                        ->relationship('user', 'name')
                                        ->helperText('Select the user that belongs to this project')
                                        ->required()
                                        ->searchable()
                                        ->preload() : null,
                                    // Dashboard-only fields
                                    $panel === 'dashboard' ? Hidden::make('project_status') : null,
                                    // Common fields
                                    Select::make('types')
                                        ->label('Project Categories')
                                        ->relationship('types', 'type_name')
                                        ->multiple()
                                        ->preload()
                                        ->searchable()
                                        ->createOptionForm([
                                            TextInput::make('type_name')
                                                ->required()
                                                ->maxLength(255)
                                                ->unique('project_types', 'type_name')
                                                ->helperText('Enter the name for the project type'),
                                        ])
                                        ->createOptionAction(
                                            fn(FormAction $action) => $action
                                                ->modalWidth('sm')
                                                ->modalHeading('Create Project Type')
                                        ),
                                    Select::make('completed_status')
                                        ->label('Project Status')
                                        ->options(ProjectCompletedStatus::class)
                                        ->default(ProjectCompletedStatus::InProgress->value)
                                        ->live(),
                                    DatePicker::make('completion_date')
                                        ->label('Completion Date')
                                        ->default(now()->format('Y-m-d'))
                                        ->format('Y-m-d')
                                        ->live()
                                        ->visible(function (Get $get, ?Livewire $livewire) {
                                            $selectedStatus = $get('completed_status');
                                            if ($livewire && !empty($livewire->data['completed_status']->value ?? null)) {
                                                $selectedStatus = $livewire->data['completed_status']->value;
                                            }
                                            return $selectedStatus === ProjectCompletedStatus::Completed->value;
                                        }),
                                    TextInput::make('project_cost')
                                        ->label('Project Cost')
                                        ->numeric()
                                        ->validationMessages([
                                            'numeric' => 'The project cost must be a number.',
                                        ])
                                        ->default(0)
                                        ->live()
                                        ->visible(function (Get $get, ?Livewire $livewire) {
                                            $selectedStatus = $get('completed_status');
                                            if ($livewire && !empty($livewire->data['completed_status']->value ?? null)) {
                                                $selectedStatus = $livewire->data['completed_status']->value;
                                            }
                                            return $selectedStatus === ProjectCompletedStatus::Completed->value;
                                        }),
                                    TagsInput::make('tags')
                                        ->label('Tags')
                                        ->rules(['max:15'])
                                        ->placeholder('Enter the tags for the project'),
                                ])),
                            Section::make('Media')
                                ->schema([
                                    SpatieMediaLibraryFileUpload::make('featured_image')
                                        ->collection('featured_image')
                                        ->image()
                                        ->imageEditor()
                                        ->preserveFilenames()
                                        ->required()
                                        ->maxSize(5120)
                                        ->label('Featured Image'),
                                ]),
                        ]),
                ])
        ];
    }
}
