<?php

namespace App\Filament\Resources\Project\ProjectTypeResource\Pages;

use App\Filament\Resources\Project\ProjectTypeResource;
use Filament\Resources\Pages\Page;
use Filament\Resources\Pages\Concerns\InteractsWithRecord;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Filters\Filter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Notifications\Notification;
use Filament\Tables\Contracts\HasTable;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use App\Models\Project\ProjectType;
use Illuminate\Support\Str; 
use Filament\Forms\Form;
use Filament\Forms\Set;
use Filament\Forms\Get;
use Illuminate\Support\Facades\DB;


class ProjectTypes extends Page implements HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;

    protected static string $resource = ProjectTypeResource::class;

    protected static string $view = 'filament.resources.project.project-type-resource.pages.project-types';

    public ?array $data = [];
    public ?ProjectType $projectType = null;

    public function mount(): void
    {
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('type_name')
                    ->label('Title')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (Get $get, Set $set, ?string $state) {
                        if (!$this->projectType && empty($get('slug'))) {
                            $set('slug', Str::slug($state));
                        }
                    }),
                TextInput::make('slug')
                    ->label('Slug')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (?string $state, Set $set) {
                        if ($state) {
                            $set('slug', Str::slug($state));
                        }
                    })
                    ->dehydrated()
                    ->unique(ignoreRecord: true),
                RichEditor::make('type_description')
                    ->disableToolbarButtons([
                        'codeBlock',
                    ])
                    ->label('Description')
            ])
            ->statePath('data');
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(ProjectType::query())
            ->columns([
                TextColumn::make('type_name')
                    ->label('Title')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('slug')
                    ->searchable()
                    ->wrap()
                    ->sortable(),
                TextColumn::make('projects_count')
                    ->label('Count')
                    ->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                TrashedFilter::make(),
            ])
            ->actions([
                ActionGroup::make([
                    
                    Action::make('edit')
                        ->icon('heroicon-m-pencil-square')
                        ->hidden(fn(ProjectType $record): bool => $record->trashed())
                        ->action(function (ProjectType $record): void {
                            $this->projectType = $record;
                            $this->data = $record->toArray();
                        }),
                    ReplicateAction::make()
                        ->excludeAttributes(['projects_count'])        
                        ->hidden(fn(ProjectType $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(ProjectType $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                $query->select('project_types.*')
                    ->selectSub(
                        function ($subQuery) {
                            $subQuery->from('project_project_type')
                                    ->selectRaw('COUNT(*)')
                                    ->whereColumn('project_project_type.project_type_id', 'project_types.id');
                        },
                        'projects_count'
                    );
                return $query;
            });
    }

    public function save(): void
    {
        $this->validate();

        try {
            DB::beginTransaction();

            $data = $this->form->getState();

            if ($this->projectType) {
                $this->projectType->update($data);
                $message = 'Project Type updated successfully';
            } else {
                ProjectType::create($data);
                $message = 'Project Type created successfully';
            }

            DB::commit();

            $this->projectType = null;
            $this->data = [];
            $this->form->fill();

            Notification::make()
                ->success()
                ->title($message)
                ->send();
        } catch (\Exception $e) {
            DB::rollBack();
            
            Notification::make()
                ->danger()
                ->title('Error saving tag')
                ->body('Please try again.')
                ->send();
        }
    }

    public function cancelEdit(): void
    {
        $this->projectType = null;
        $this->data = [];
        $this->form->fill();
    }
}
