<?php

namespace App\Filament\Resources\Review\Trait;

use Filament\Forms\Form;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use App\Enums\MemberRatingType;
use App\Models\User\User;

trait ReviewFormTrait
{
    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 12,
                    ])
                    ->schema([
                        Group::make()
                            ->columnSpan(['lg' => 8])
                            ->schema([
                                Section::make('Review Information')
                                    ->schema([
                                        TextInput::make('review_title')
                                            ->label('Review Title')
                                            ->required(),
                                        TextInput::make('review_content')
                                            ->label('Review Description')
                                            ->required(),
                                        TextInput::make('review_rating')
                                            ->label('Rating Value')
                                            ->numeric()
                                            ->minValue(1)
                                            ->maxValue(5)
                                            ->required(),
                                    ]),
                                Section::make('Rating Details')
                                    ->schema([
                                        Repeater::make('ratings')
                                            ->relationship()
                                            ->schema([
                                                Grid::make(2)
                                                    ->schema([
                                                        Select::make('rating_type')
                                                            ->label('Rating Type')
                                                            ->options(collect(MemberRatingType::cases())->mapWithKeys(function ($case) {
                                                                return [$case->value => $case->getLabel()];
                                                            }))
                                                            ->required()
                                                            ->disabled()
                                                            ->dehydrated(),
                                                        TextInput::make('rating')
                                                            ->label('Rating')
                                                            ->numeric()
                                                            ->minValue(1)
                                                            ->maxValue(5)
                                                            ->required(),
                                                    ]),
                                            ])
                                            ->defaultItems(0)
                                            ->minItems(count(MemberRatingType::cases()))
                                            ->maxItems(count(MemberRatingType::cases()))
                                            ->reorderable(false)
                                            ->deletable(false)
                                            ->addable(false)
                                            ->afterStateHydrated(function ($component, $state, $record) {
                                                if (!$record) {
                                                    // For new records, create default items for each rating type
                                                    $defaultItems = collect(MemberRatingType::cases())->map(function ($case) {
                                                        return [
                                                            'rating_type' => $case->value,
                                                            'rating' => null,
                                                        ];
                                                    })->toArray();
                                                    $component->state($defaultItems);
                                                }
                                            })
                                            ->itemLabel(fn (array $state): ?string => 
                                                isset($state['rating_type']) 
                                                    ? MemberRatingType::from($state['rating_type'])->getLabel()
                                                    : null
                                            ),
                                    ]),
                            ]),
                        Group::make()
                            ->columnSpan(['lg' => 4])
                            ->schema([
                                Section::make('Review Status')
                                    ->schema([
                                        Select::make('review_user_id')
                                            ->label("Returned By")
                                            ->relationship('user', 'name')
                                            ->required()
                                            ->searchable()
                                            ->preload(),

                                        Select::make('reviewable_id')
                                            ->label("Review For")
                                            ->options(function () {
                                                return User::role('Service')
                                                    ->with('profile')
                                                    ->get()
                                                    ->pluck('name', 'id')
                                                    ->toArray();
                                            })
                                            ->searchable()
                                            ->preload()
                                            ->required()
                                            ->reactive()
                                            ->afterStateUpdated(function ($state, $set) {
                                                if ($state) {
                                                    $set('reviewable_type', User::class);
                                                }
                                            }),

                                        Hidden::make('reviewable_type')
                                            ->default(User::class),
                                            
                                        Toggle::make('status')
                                            ->label("Review Status")
                                            ->required(),
                                    ]),
                            ]),
                    ]),
            ]);
    }
}
