<?php

namespace App\Filament\Resources\Team;

use App\Filament\Resources\Team\TeamMemberResource\Pages;
use App\Filament\Resources\Team\TeamMemberResource\RelationManagers;
use App\Models\Team\TeamMember;
use App\Models\User\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Toggle;
use Illuminate\Support\Facades\Auth;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Repeater;
use Filament\Tables\Actions\ActionGroup;
use App\Enums\TeamMemberStatus;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use App\Forms\Components\TinyMceEditor;

class TeamMemberResource extends Resource
{
    protected static ?string $model = TeamMember::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';

    protected static ?string $navigationGroup = 'Team Management';

    protected static ?string $navigationLabel = 'Team Members';

    protected static ?int $navigationSort = 1;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 10,
                    ])
                    ->schema([
                        // Main Content Column (7/10 = 70%)
                        Group::make()
                            ->columnSpan(['lg' => 7])
                            ->schema([
                                Section::make('Basic Information')
                                    ->description('Enter the team member\'s basic details')
                                    ->schema([
                                        TextInput::make('first_name')
                                            ->placeholder('Enter the team member\'s first name')
                                            ->required()
                                            ->maxLength(100),
                                        TextInput::make('last_name')
                                            ->placeholder('Enter the team member\'s last name')
                                            ->maxLength(100),
                                        TextInput::make('email')
                                            ->email()
                                            ->placeholder('Enter the team member\'s email')
                                            ->maxLength(100),
                                        TextInput::make('phone')
                                            ->tel()
                                            ->mask('(999) 999-9999')
                                            ->placeholder('Enter the team member\'s phone number')
                                            ->maxLength(100),
                                        TextInput::make('experience_years')
                                            ->placeholder('Enter the team member\'s experience years')
                                            ->maxLength(100)
                                            ->regex('/^[0-9]+$/'),
                                    ])->columns(2),

                                Section::make('Professional Details')
                                    ->schema([
                                        Repeater::make('credentials')
                                            ->label('Credentials & Certifications')
                                            ->schema([
                                                TextInput::make('title')
                                                    ->label('Credential Title')
                                                    ->required()
                                                    ->maxLength(20)
                                                    ->placeholder('e.g., Water Treatment Operator'),
                                                TextInput::make('description')
                                                    ->label('Description')
                                                    ->required()
                                                    ->maxLength(100)
                                                    ->placeholder('e.g., Certified in 2023 by ABC Organization'),
                                            ])
                                            ->columns(1)
                                            ->defaultItems(0)
                                            ->addActionLabel('Add Credential')
                                            ->reorderableWithButtons()
                                            ->collapsible()
                                            ->itemLabel(fn(array $state): ?string => $state['title'] ?? null)
                                            ->columnSpanFull(),
                                        TinyMceEditor::make('description')
                                            ->plugins('lists link autolink autoresize')
                                            ->toolbar('bold italic underline | bullist numlist | undo redo')
                                            ->menubar(false)
                                            ->branding(false)
                                            ->promotion(false)
                                            ->height(300)
                                            ->placeholder('Enter the team member\'s description')
                                            ->columnSpanFull(),
                                    ]),
                            ]),

                        // Sidebar Column (3/10 = 30%)
                        Group::make()
                            ->columnSpan(['lg' => 3])
                            ->schema([
                                Section::make('Settings')
                                    ->schema([
                                        Select::make('team_member_status')
                                            ->label('Status')
                                            ->options([
                                                TeamMemberStatus::Draft->value => TeamMemberStatus::Draft->name,
                                                TeamMemberStatus::Published->value => TeamMemberStatus::Published->name,
                                                TeamMemberStatus::Pending->value => TeamMemberStatus::Pending->name,
                                                TeamMemberStatus::Expired->value => TeamMemberStatus::Expired->name,
                                            ])
                                            ->live()
                                            ->default(TeamMemberStatus::Draft->value)
                                            ->required(),
                                        Select::make('user_id')
                                            ->label('Belongs To')
                                            ->relationship('user', 'name')
                                            ->helperText('Select the user that belongs to this team member')
                                            ->required()
                                            ->searchable()
                                            ->preload(),
                                        TextInput::make('team_member_role')
                                            ->placeholder('Enter the team member\'s role')
                                            ->maxLength(100),
                                    ]),
                                Section::make('Media')
                                    ->schema([
                                        SpatieMediaLibraryFileUpload::make('featured_image')
                                            ->collection('featured_image')
                                            ->imageEditor()
                                            ->required()
                                            ->hintIcon('heroicon-m-photo')
                                            ->conversion('thumb'),
                                    ]),

                                Section::make('Contact & Social')
                                    ->schema([
                                        Textarea::make('address')
                                            ->placeholder('Enter the team member\'s address')
                                            ->rows(3),
                                        TextInput::make('social_links.facebook')
                                            ->label('Facebook')
                                            ->url()
                                            ->prefixIcon('fab-facebook')
                                            ->placeholder('https://facebook.com/username'),
                                        TextInput::make('social_links.linkedin')
                                            ->label('LinkedIn')
                                            ->url()
                                            ->prefixIcon('fab-linkedin')
                                            ->placeholder('https://linkedin.com/in/username'),
                                        TextInput::make('social_links.youtube')
                                            ->label('YouTube')
                                            ->url()
                                            ->prefixIcon('fab-youtube')
                                            ->placeholder('https://youtube.com/@username'),
                                        TextInput::make('social_links.instagram')
                                            ->label('Instagram')
                                            ->url()
                                            ->prefixIcon('fab-instagram')
                                            ->placeholder('https://instagram.com/username'),
                                        TextInput::make('social_links.x')
                                            ->label('X (Twitter)')
                                            ->url()
                                            ->prefixIcon('fab-x-twitter')
                                            ->placeholder('https://x.com/username'),
                                    ]),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                TextColumn::make('first_name')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('last_name')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('email')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('phone')
                    ->searchable(),
                TextColumn::make('experience_years')
                    ->label('Experience')
                    ->suffix(' Years')
                    ->sortable(),
                TextColumn::make('user.name')
                    ->label('Belongs To')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('team_member_status')
                    ->badge()
                    ->label('Status'),
                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\SelectFilter::make('team_member_status')
                    ->options([
                        TeamMemberStatus::Draft->value => TeamMemberStatus::Draft->name,
                        TeamMemberStatus::Published->value => TeamMemberStatus::Published->name,
                        TeamMemberStatus::Pending->value => TeamMemberStatus::Pending->name,
                        TeamMemberStatus::Expired->value => TeamMemberStatus::Expired->name,
                    ])
                    ->label('Status'),
                Tables\Filters\SelectFilter::make('user_id')
                    ->relationship('user', 'name')
                    ->label('Belongs To')
                    ->preload()
                    ->searchable(),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)

            ->actions([
                ActionGroup::make([

                    Action::make('approve')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(
                            fn(TeamMember $record): bool =>
                            $record->team_member_status->value === TeamMemberStatus::Pending->value
                        )
                        ->action(function (TeamMember $record): void {
                            $record->team_member_status = TeamMemberStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Team Member')
                        ->modalDescription('Are you sure you want to approve this team member? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, approve')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Team Member Approved')
                                ->body('Team Member has been approved successfully')
                        ),
                    Action::make('publish')
                        ->icon('heroicon-o-arrow-up-circle')
                        ->color('success')
                        ->visible(
                            fn(TeamMember $record): bool =>
                            in_array($record->team_member_status->value, [TeamMemberStatus::Draft->value, TeamMemberStatus::Expired->value])
                        )
                        ->action(function (TeamMember $record): void {
                            $record->team_member_status = TeamMemberStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Publish Team Member')
                        ->modalDescription('Are you sure you want to publish this team member? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, Publish')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Team Member Published')
                                ->body('Team Member has been published successfully')
                        ),
                    ViewAction::make()
                        ->slideOver()
                        ->hidden(fn(TeamMember $record): bool => $record->trashed()),
                    EditAction::make()
                        ->hidden(fn(TeamMember $record): bool => $record->trashed()),
                    ReplicateAction::make()
                        ->before(function (TeamMember $record): void {
                            $record->team_member_status = TeamMemberStatus::Draft->value;
                        })
                        ->hidden(fn(TeamMember $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(TeamMember $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTeamMembers::route('/'),
            'create' => Pages\CreateTeamMember::route('/create'),
            'edit' => Pages\EditTeamMember::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
