<?php

namespace App\Filament\Resources\Team\TeamMemberResource\Pages;

use App\Filament\Resources\Team\TeamMemberResource;
use Filament\Actions;
use Filament\Resources\Pages\CreateRecord;
use Illuminate\Contracts\View\View;
use Filament\Notifications\Notification;
use App\Enums\TeamMemberStatus;

class CreateTeamMember extends CreateRecord
{
    protected static string $resource = TeamMemberResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('publish')
                ->label(fn () => match($this->data['team_member_status'] ?? null) {
                    TeamMemberStatus::Pending->value => 'Save',
                    default => 'Publish',
                })
                ->icon(fn () => match($this->data['team_member_status'] ?? null) {
                    TeamMemberStatus::Pending->value => 'heroicon-o-document-text',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn () => match($this->data['team_member_status'] ?? null) {
                    TeamMemberStatus::Pending->value => 'primary',
                    default => 'success',
                })
                ->action(function () {
                    $data = $this->form->getState();
                    
                    if ($data['team_member_status'] === TeamMemberStatus::Draft->value || $data['team_member_status'] === TeamMemberStatus::Published->value) {
                        $this->data['team_member_status'] = TeamMemberStatus::Published->value;
                        $this->data['published_at'] = now();
                    }
                    
                    $this->create();
                })
                ->keyBindings(['mod+s']),

            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['team_member_status'] = TeamMemberStatus::Draft->value;
                    $this->create();
                }),

            Actions\Action::make('cancel')
                ->label('Cancel')
                ->icon('heroicon-o-x-mark')
                ->color('gray')
                ->url(fn () => TeamMemberResource::getUrl()),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            Actions\Action::make('create')
                ->label(fn () => match($this->data['team_member_status'] ?? null) {
                    TeamMemberStatus::Pending->value => 'Save',
                    default => 'Publish',
                })
                ->icon(fn () => match($this->data['team_member_status'] ?? null) {
                    TeamMemberStatus::Pending->value => 'heroicon-o-document-text',
                    default => 'heroicon-o-arrow-up-circle',
                })
                ->color(fn () => match($this->data['team_member_status'] ?? null) {
                    TeamMemberStatus::Pending->value => 'primary',
                    default => 'success',
                })
                ->action(function () {
                    if ($this->data['team_member_status'] === TeamMemberStatus::Draft->value || $this->data['team_member_status'] === TeamMemberStatus::Published->value) {
                        $this->data['team_member_status'] = TeamMemberStatus::Published->value;
                        $this->data['published_at'] = now();
                    }
                    
                    $this->create();
                })
                ->keyBindings(['mod+s']),
            Actions\Action::make('saveDraft')
                ->label('Save Draft')
                ->icon('heroicon-o-document')
                ->color('gray')
                ->action(function () {
                    $this->data['team_member_status'] = TeamMemberStatus::Draft->value;
                    $this->create();
                }),
            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        // If the post status is set to Published, set the published_at timestamp to now
        if ($data['team_member_status'] === TeamMemberStatus::Published->value) {
            $data['published_at'] = now();
        }

        return $data;
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('edit', ['record' => $this->record]);
    }

    protected function getCreatedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match($this->data['team_member_status']) {
                TeamMemberStatus::Draft->value => 'Draft Saved',
                TeamMemberStatus::Pending->value => 'Team Member Saved',
                default => 'Team Member Published',
            })
            ->body(match($this->data['team_member_status']) {
                TeamMemberStatus::Draft->value => 'Your team member has been saved as draft.',
                TeamMemberStatus::Pending->value => 'Your team member has been saved as pending.',
                default => 'Your team member has been published successfully.',
            });
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
