<?php

namespace App\Filament\Resources\Team\TeamMemberResource\Pages;

use App\Filament\Resources\Team\TeamMemberResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use Filament\Notifications\Notification;
use App\Enums\TeamMemberStatus;
use Illuminate\Contracts\View\View;

class EditTeamMember extends EditRecord
{
    protected static string $resource = TeamMemberResource::class;

    protected function getSavedNotification(): ?Notification
    {
        return Notification::make()
            ->success()
            ->title(match($this->data['team_member_status']) {
                TeamMemberStatus::Draft->value => 'Draft Updated',
                TeamMemberStatus::Pending->value => 'Team Member Updated',
                default => 'Team Member Updated',
            })
            ->body(match($this->data['team_member_status']) {
                TeamMemberStatus::Draft->value => 'Your team member has been updated and saved as draft.',
                TeamMemberStatus::Pending->value => 'Your team member has been updated and saved as pending.',
                default => 'Your team member has been updated successfully.',
            });
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('save')
                ->label('Update')
                ->action('save')
                ->icon('heroicon-o-check-circle')
                ->color('primary')
                ->keyBindings(['mod+s']),

            

            Actions\ActionGroup::make([
                Actions\ActionGroup::make([
                    // Array of action

                Actions\DeleteAction::make()
                    ->label('Move to Trash')
                    ->icon('heroicon-o-trash')
                    ->requiresConfirmation()
                    ->hidden(fn (): bool => $this->record->trashed())
                    ->modalHeading('Delete Team Member')
                    ->modalDescription('Are you sure you want to delete this team member? This action can be undone later.')
                    ->modalSubmitActionLabel('Yes, delete it')
                    ->successNotification(
                        Notification::make()
                            ->success()
                            ->title('Team Member deleted')
                            ->body('The team member has been moved to trash.')
                    ),
                ])->dropdown(false),
                Actions\Action::make('cancel')
                    ->label('Cancel')
                    ->icon('heroicon-o-x-mark')
                    ->color('gray')
                    ->url(fn () => TeamMemberResource::getUrl()),
            ])
            ->tooltip('More Actions')
            ->icon('heroicon-m-ellipsis-vertical')
            ->color('info')
            ->button()
            ->extraAttributes([
                'class' => 'more-actions-btn',
            ]),
        ];
    }

    protected function afterDelete(): void
    {
        if (! $this->record->trashed()) {
            $this->record->update(['team_member_status' => TeamMemberStatus::Trashed]);
        }
    }

    protected function afterRestore(): void
    {
        $this->record->update(['team_member_status' => TeamMemberStatus::Draft]);
    }

    protected function afterSave(): void
    {
        if ($this->record->team_member_status !== TeamMemberStatus::Trashed && $this->record->trashed()) {
            $this->record->restore();
        }
    }

    protected function getFormActions(): array
    {
        return [
            $this->getSaveFormAction()
                ->label('Update')
                ->icon('heroicon-o-check-circle'),
            
            $this->getCancelFormAction()
                ->icon('heroicon-o-x-mark'),
        ];
    }

    public function getHeader(): ?View
    {
        return view('filament.pages.header', [
            'actions' => $this->getHeaderActions(),
        ]);
    }
}
