<?php

namespace App\Filament\Resources\User\PermissionResource\Pages;

use App\Filament\Resources\User\PermissionResource;
use Filament\Resources\Pages\Page;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Table;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Tables\Columns\TextColumn;
use App\Models\User\MyPermission;
use Filament\Tables\Actions\Action;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Notifications\Notification;
class PermissionsPage extends Page implements HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;


    protected static string $resource = PermissionResource::class;

    protected static string $view = 'filament.resources.user.permission-resource.pages.permissions-page';

    public ?array $data = [];

    public $selectedPermission = null;

    public function mount(): void
    {
        $this->form->fill();
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(MyPermission::query())
            ->columns([
                TextColumn::make('name')
                    ->searchable(),
                TextColumn::make('parent.name')
                    ->label('Parent')
                    ->formatStateUsing(fn ($state) => !empty($state) ? $state : '--'),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('edit')
                        ->icon('heroicon-m-pencil-square')
                        ->action(function (MyPermission $record): void {
                            $this->selectedPermission = $record;
                            $this->data = $record->toArray();
                        }),
                    DeleteAction::make()
                        ->icon('heroicon-m-trash'),
                ])
                ->tooltip('Actions')
                ->iconButton()
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->deselectRecordsAfterCompletion()
                ])
            ]);
    }

    public function form(Form $form): Form
    {
        return $form->schema([
            TextInput::make('name')
                ->required()
                ->maxLength(255),
            Select::make('parent_id')
                ->label('Parent')
                ->options(MyPermission::where('parent_id', null)->get()->pluck('name', 'id')),
            Select::make('guard_name')
                ->options([
                    'web' => 'Web',
                    'api' => 'API',
                ])
                ->required(),
        ])
        ->statePath('data');
    }

    public function save(): void
    {
        $data = $this->form->getState();

        if ($this->selectedPermission) {
            // Update existing tag
            $this->selectedPermission->update($data);
            $message = 'Permission updated successfully';
        } else {
            // Create new tag
            MyPermission::create($data);
            $message = 'Permission created successfully';
        }

        $this->selectedPermission = null;
        $this->data = [];
        $this->form->fill();

        Notification::make()
            ->success()
            ->title($message)
            ->send();
    }

    public function cancelEdit(): void
    {
        $this->selectedPermission = null;
        $this->data = [];
        $this->form->fill();
    }
}
