<?php

namespace App\Filament\Resources\User;

use App\Filament\Resources\User\UserResource\Pages;
use App\Filament\Resources\User\UserResource\RelationManagers;
use App\Models\User\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use App\Enums\UserProfileType;
use Filament\Forms\Components\Select;
use App\Enums\UserProfileStatus;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Get;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use App\Models\User\MyRole;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\RichEditor;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make()
                    ->columns([
                        'default' => 1,
                        'lg' => 12, // Using 12-column grid
                    ])
                    ->schema([
                        Group::make()
                            ->columnSpan(['lg' => 8])
                            ->schema([
                                Section::make('Personal Information')
                                    ->schema([
                                        TextInput::make('name')
                                            ->label('Full Name')
                                            ->hidden()
                                            ->dehydrated()
                                            ->live()
                                            ->afterStateUpdated(function (string $operation, $state, Get $get) {
                                                return $get('profile.first_name') . ' ' . $get('profile.last_name');
                                            }),
                                        TextInput::make('profile.first_name')
                                            ->label('First Name')
                                            ->required()
                                            ->live()
                                            ->maxLength(255),
                                        TextInput::make('profile.last_name')
                                            ->label('Last Name')
                                            ->required()
                                            ->maxLength(255),
                                        TextInput::make('email')
                                            ->label('Email')
                                            ->email()
                                            ->required()
                                            ->maxLength(255),
                                        TextInput::make('profile.phone')
                                            ->label('Phone')
                                            ->tel()
                                            ->mask('(999) 999-9999')
                                            //->required()
                                            ->maxLength(255),
                                        TextInput::make('profile.fax')
                                            ->label('Fax')
                                            ->mask('(999) 999-9999')
                                            ->tel()
                                            ->maxLength(255),
                                    ])
                                    ->columns(['lg' => 2]),
                                Section::make('Company Information')
                                    ->schema([
                                        TextInput::make('profile.company_name')
                                            ->label('Company Name')
                                            //->required()
                                            ->maxLength(255),
                                        TextInput::make('profile.website')
                                            ->label('Company Website')
                                            ->url()
                                            //->required()
                                            ->maxLength(255),
                                        TextInput::make('profile.year_established')
                                            ->label('Year Established')
                                            ->required()
                                            ->numeric(),
                                        RichEditor::make('profile.about')
                                            ->label('About Company'),
                                        //->required(),
                                        Textarea::make('profile.business_hours')
                                            ->label('Business Hours'),
                                        //->required(),
                                        Textarea::make('profile.slogan')
                                            ->label('Slogan'),
                                        Textarea::make('profile.services')
                                            ->label('Services'),
                                        Textarea::make('profile.other_services')
                                            ->label('Other Services'),
                                    ])
                                    ->visible(fn(Get $get) => $get('profile.profile_type') === UserProfileType::Business->value),
                                Section::make('Social Media')
                                    ->schema([
                                        TextInput::make('profile.social_media_links.facebook')
                                            ->inlineLabel(true)
                                            ->label('Facebook')
                                            ->url()
                                            ->prefixIcon('fab-facebook'),
                                        TextInput::make('profile.social_media_links.instagram')
                                            ->inlineLabel(true)
                                            ->label('Instagram')
                                            ->url()
                                            ->prefixIcon('fab-instagram'),
                                        TextInput::make('profile.social_media_links.twitter')
                                            ->inlineLabel(true)
                                            ->label('Twitter')
                                            ->url()
                                            ->prefixIcon('fab-twitter'),
                                        TextInput::make('profile.social_media_links.linkedin')
                                            ->inlineLabel(true)
                                            ->label('LinkedIn')
                                            ->url()
                                            ->prefixIcon('fab-linkedin'),
                                        TextInput::make('profile.social_media_links.youtube')
                                            ->inlineLabel(true)
                                            ->label('Youtube')
                                            ->url()
                                            ->prefixIcon('fab-youtube'),
                                        TextInput::make('profile.social_media_links.tiktok')
                                            ->inlineLabel(true)
                                            ->label('TikTok')
                                            ->url()
                                            ->prefixIcon('fab-tiktok'),
                                        TextInput::make('profile.social_media_links.pinterest')
                                            ->inlineLabel(true)
                                            ->label('Pinterest')
                                            ->url()
                                            ->prefixIcon('fab-pinterest'),
                                    ]),
                                Section::make('Password')
                                    ->columns(['lg' => 2])
                                    ->schema([
                                        TextInput::make('password')
                                            ->label('Password')
                                            ->password()
                                            ->live()
                                            ->required(fn(string $operation): bool => $operation === 'create')
                                            ->dehydrated(fn($state) => filled($state))
                                            ->maxLength(255),
                                        TextInput::make('password_confirmation')
                                            ->label('Confirm Password')
                                            ->password()
                                            ->required(fn(Get $get): bool => filled($get('password')))
                                            ->maxLength(255),
                                        Checkbox::make('password_change_email')
                                            ->label('Send new password email to user.')
                                            ->default(false)
                                            ->hidden(fn(Get $get): bool => !filled($get('password'))),
                                    ]),
                            ]),
                        Group::make()
                            ->columnSpan(['lg' => 4])
                            ->schema([
                                Section::make('Avatar')
                                    ->schema([
                                        SpatieMediaLibraryFileUpload::make('avatar')
                                            ->collection('avatar')
                                            ->label('Upload Avatar'),
                                        Select::make('profile.profile_type')
                                            ->options(UserProfileType::class)
                                            ->required()
                                            ->default(UserProfileType::Personal->value)
                                            ->live()
                                            ->native(false),
                                        Select::make('profile.status')
                                            ->options(UserProfileStatus::class)
                                            ->required()
                                            ->default(UserProfileStatus::Active->value)
                                            ->native(false),
                                        Select::make('roles')
                                            ->relationship('roles', 'name')
                                            ->native(false)
                                            ->required(),
                                    ])
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('email')
                    ->searchable(),
                Tables\Columns\IconColumn::make('email_verified_at')
                    ->label('Verified')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-badge')
                    ->falseIcon('heroicon-o-x-mark'),
                Tables\Columns\TextColumn::make('roles.name')
                    ->label('Role')
                    ->badge()
                    ->searchable(),
                Tables\Columns\TextColumn::make('profile.status')
                    ->label('Status')
                    ->badge()
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime('d-m-Y'),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\EditAction::make(),
                    Tables\Actions\Action::make('send_verification_email')
                        ->label('Send Verification Email')
                        ->icon('heroicon-o-envelope')
                        ->color('info')
                        ->visible(fn($record) => !$record->hasVerifiedEmail())
                        ->action(function ($record) {
                            event(new \Illuminate\Auth\Events\Registered($record));
                            \Filament\Notifications\Notification::make()
                                ->title('Verification email sent successfully')
                                ->success()
                                ->send();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Send Verification Email')
                        ->modalDescription('Are you sure you want to send a verification email to this user?'),
                    Tables\Actions\Action::make('deactivate_user')
                        ->label('Deactivate User')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->visible(fn($record) => $record->profile && $record->profile->status === \App\Enums\UserProfileStatus::Active)
                        ->action(function ($record) {
                            $record->profile->update(['status' => \App\Enums\UserProfileStatus::NotActive]);
                            \Filament\Notifications\Notification::make()
                                ->title('User deactivated successfully')
                                ->success()
                                ->send();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Deactivate User')
                        ->modalDescription('Are you sure you want to deactivate this user? This will set their profile status to not active.'),
                    Tables\Actions\Action::make('activate_user')
                        ->label('Activate User')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(fn($record) => $record->hasVerifiedEmail() && $record->profile && $record->profile->status === \App\Enums\UserProfileStatus::NotActive)
                        ->action(function ($record) {
                            $record->profile->update(['status' => \App\Enums\UserProfileStatus::Active]);
                            \Filament\Notifications\Notification::make()
                                ->title('User activated successfully')
                                ->success()
                                ->send();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Activate User')
                        ->modalDescription('Are you sure you want to activate this user? This will set their profile status to active.'),
                ]),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
