<?php
namespace App\Filament\TipTapEditor\TiptapBlock;

use Faker\Core\Color;
use FilamentTiptapEditor\TiptapBlock;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Toggle;
use Illuminate\Support\Facades\Cache;

class ImageAdBlock extends TiptapBlock
{
    public string $preview = 'tiptap-editor.tiptap-blocks.previews.image-ad-block-preview';
    public string $rendered = 'tiptap-editor.tiptap-blocks.rendered.image-ad-block-render';
    public string $width = '2xl';
    public bool $slideOver = true;
    public ?string $icon = 'heroicon-o-photo';
    
    protected static ?string $heading = 'Image Advertisement';

    public function getFormSchema(): array
    {
        return [
            Section::make('Advertisement Details')
                ->description('Configure the advertisement settings and appearance')
                ->compact()
                ->columns(2)
                ->schema([
                    TextInput::make('thumbnaillink')
                        ->label('Destination URL')
                        ->required()
                        ->url()
                        ->prefix('https://')
                        ->placeholder('Enter destination URL')
                        ->columnSpan(1)
                        ->helperText('The URL where users will be directed when clicking the ad'),

                    Select::make('buttontarget')
                        ->label('Link Target')
                        ->default('_self')
                        ->options([
                            '_self' => 'Same Window',
                            '_blank' => 'New Window',
                        ])
                        ->columnSpan(1)
                        ->helperText('Choose how the link should open'),

                    FileUpload::make('imageurl')
                        ->label('Advertisement Image')
                        ->image()
                        ->required()
                        ->imageResizeMode('cover')
                        ->imageCropAspectRatio('2.35:1')
                        ->imageResizeTargetWidth('1200')
                        ->imageResizeTargetHeight('510')
                        ->imageResizeUpscale(false)
                        ->maxSize(5120) // 5MB
                        ->acceptedFileTypes(['image/jpeg', 'image/png', 'image/webp'])
                        ->columnSpan(2),
                        //->helperText('Recommended size: 1200x510px. Max size: 5MB'),

                    TextInput::make('alttext')
                        ->label('Alt Text')
                        ->required()
                        ->maxLength(100)
                        ->placeholder('Descriptive text for the image')
                        ->columnSpan(1)
                        ->helperText('Provide descriptive text for accessibility'),

                    TextInput::make('trackingid')
                        ->label('Tracking ID')
                        ->placeholder('Optional tracking identifier')
                        ->maxLength(50)
                        ->columnSpan(1)
                        ->helperText('Add a custom tracking ID for analytics'),
                ]),

            Section::make('Display Options')
                ->description('Configure how the advertisement is displayed')
                ->compact()
                ->columns(2)
                ->schema([
                    Toggle::make('lazyload')
                        ->label('Enable Lazy Loading')
                        ->default(true)
                        ->columnSpan(1)
                        ->helperText('Load image only when it enters viewport'),

                    Toggle::make('responsive')
                        ->label('Enable Responsive Sizing')
                        ->default(true)
                        ->columnSpan(1)
                        ->helperText('Automatically adjust image size for different screens'),
                ]),
        ];
    }

    protected function afterCreate(): void
    {
        $imageUrl = $this->state['imageurl'] ?? null;
        if ($imageUrl) {
            $cacheKey = 'thumbnail-ad-url-' . md5($imageUrl);
            Cache::forget($cacheKey);
        }
    }

    protected function afterUpdate(): void
    {
        $imageUrl = $this->state['imageurl'] ?? null;
        if ($imageUrl) {
            $cacheKey = 'thumbnail-ad-url-' . md5($imageUrl);
            Cache::forget($cacheKey);
        }
    }
}