<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use App\Models\User\MemberCategory;
use App\Policies\Users\MemberCategoryPolicy;

class DashboardSidebar
{
    /**
     * Get all menu items from config
     *
     * @return array
     */
    public static function getMenuItems(): array
    {
        return config('dashboard-sidebar.menu_items', []);
    }

    /**
     * Check if the current route matches the menu item
     *
     * @param array $item
     * @return bool
     */
    public static function isActive(array $item): bool
    {
        if (!isset($item['route'])) {
            return false;
        }

        $currentRoute = Route::currentRouteName();

        // For dropdown menus, check if any submenu item is active
        if (isset($item['has_dropdown']) && $item['has_dropdown'] && isset($item['submenu'])) {
            foreach ($item['submenu'] as $submenuItem) {
                if (isset($submenuItem['route']) && $submenuItem['route'] === $currentRoute) {
                    return true;
                }
            }
            return false;
        }

        // For regular menu items
        if (isset($item['exact']) && $item['exact']) {
            return $currentRoute === $item['route'];
        }

        // For menu items that should match route prefix
        return str_starts_with($currentRoute, $item['route']);
    }

    /**
     * Check if a submenu item is active
     *
     * @param array $submenuItem
     * @return bool
     */
    public static function isSubmenuActive(array $submenuItem): bool
    {
        if (!isset($submenuItem['route'])) {
            return false;
        }
        return Route::currentRouteName() === $submenuItem['route'];
    }

    /**
     * Get active state classes for submenu item
     *
     * @param array $submenuItem
     * @return string
     */
    public static function getSubmenuActiveClasses(array $submenuItem): string
    {
        if (!isset($submenuItem['route'])) {
            return 'text-dark-blue/[0.4]';
        }

        return self::isSubmenuActive($submenuItem)
            ? 'text-blue-1200 after:opacity-100 bg-blue-1200/[0.1]'
            : 'text-dark-blue/[0.4]';
    }

    /**
     * Check if user has permission for menu item
     *
     * @param array $item
     * @return bool
     */
    public static function hasPermission(array $item): bool
    {
        $user = Auth::user();

        // Check permission if defined
        if (isset($item['permission']) && $item['permission'] !== null) {
            if (!$user || !Gate::allows($item['permission'])) {
                return false;
            }
        }

        // Check gates if defined
        if (isset($item['gates']) && is_array($item['gates']) && !empty($item['gates'])) {
            return self::checkGates($item);
        }

        // Check policies if defined
        if (isset($item['policies']) && is_array($item['policies']) && !empty($item['policies'])) {
            return self::checkPolicies($item);
        }
        return true;
    }

    /**
     * Get the active state classes for a menu item
     *
     * @param array $item
     * @return string
     */
    public static function getActiveClasses(array $item): string
    {
        return self::isActive($item)
            ? 'text-white bg-blue-1200 shadow-4xl after:opacity-100 hover:text-white'
            : '';
    }

    public static function checkGates(array $item): bool
    {
        $gates = $item['gates'];

        foreach ($gates as $key => $value) {
            // Handle both formats: ['Service' => 'isService'] and ['isService']
            $gateToCheck = is_string($key) ? $value : $value;
            if (!Gate::allows($gateToCheck)) {
                return false;
            }
        }

        return true;
    }

    public static function checkPolicies(array $item): bool
    {
        $policies = $item['policies'];

        foreach ($policies as $policy) {
            // Handle different policy formats
            if (is_array($policy)) {
                // Format: ['model' => ModelClass, 'method' => 'view']
                $model = $policy['model'] ?? null;
                $methods = is_array($policy['method']) ? $policy['method'] : [$policy['method'] ?? null];
                $modelInstance = $policy['instance'] ?? null;

                if ($model && class_exists($model)) {
                    // If instance is provided, use it; otherwise use the class
                    $target = $modelInstance ?? $model;
                    foreach ($methods as $method) {
                        if (!Gate::allows($method, $target)) {
                            return false;
                        }
                    }
                } else {
                    return false;
                }
            } else {
                // Format: 'policy-name' (direct policy/gate name)
                if (!Gate::allows($policy)) {
                    return false;
                }
            }
        }

        return true;
    }
}
