<?php

use Illuminate\Support\Number;

if (! function_exists('esp_currency_formated')) {
    function esp_currency_formated(int | float $price, string $currency = 'USD', string $locale = 'en', int $precision = 0): string
    {
        return Number::currency($price, in: $currency ?? 'USD', locale: $locale, precision: $precision);
    }
}

if (! function_exists('esp_classified_get_condition')) {
    function esp_classified_get_condition($classified, $key = false): string
    {
        $condition = $classified->condition;
        if ($key) {
            $condition = $condition?->getLabel();
        } else {
            $condition = $condition?->value;
        }

        return $condition;
    }
}

if (! function_exists('esp_classified_get_label')) {
    function esp_classified_get_label($classified, $key = false): string
    {
        $label = $classified->label;
        if ($key) {
            $label = $label?->getLabel();
        } else {
            $label = $label?->value;
        }
        return $label;
    }
}

if (! function_exists('esp_format_date_for_js_countdown')) {
    /**
     * Format a Carbon date instance for JavaScript countdown timers
     * This ensures timezone-aware formatting that JavaScript can properly parse
     * 
     * @param \Carbon\Carbon|null $date The Carbon date instance
     * @param bool $includeTimezone Whether to include timezone offset (default: true)
     * @return string The formatted date string for JavaScript consumption
     */
    function esp_format_date_for_js_countdown(\Carbon\Carbon $date, bool $includeTimezone = true): string
    {
        if (!$date) {
            return '';
        }

        // Always work with a copy to avoid modifying the original
        $dateCopy = $date->copy();

        if ($includeTimezone) {
            // Get the user's timezone (you might want to get this from user preferences)
            $userTimezone = config('app.timezone', 'UTC');

            // Convert to user's timezone then format as ISO string
            return $dateCopy->setTimezone($userTimezone)->toISOString();
        } else {
            // Return as UTC ISO string (recommended for consistent behavior)
            return $dateCopy->utc()->toISOString();
        }
    }
}

if (! function_exists('esp_format_date_for_js_countdown_simple')) {
    /**
     * Simple format for JavaScript countdown - always returns UTC formatted for JavaScript
     * 
     * @param \Carbon\Carbon|null $date The Carbon date instance
     * @return string The formatted date string (YYYY-MM-DD HH:mm:ss in UTC)
     */
    function esp_format_date_for_js_countdown_simple(\Carbon\Carbon $date): string
    {
        if (!$date) {
            return '';
        }

        // Always return in UTC with Z suffix for proper JavaScript parsing
        return $date->utc()->format('Y-m-d\TH:i:s\Z');
    }
}

if (! function_exists('esp_google_maps_location_link')) {
    /**
     * Generate a Google Maps link that opens the map and pins the location
     * 
     * @param float|string $lat The latitude coordinate
     * @param float|string $lng The longitude coordinate
     * @param string|null $label Optional label for the location pin
     * @return string The Google Maps URL
     */
    function esp_google_maps_location_link(float|string $lat, float|string $lng, ?string $label = null): string
    {
        // Validate coordinates
        if (empty($lat) || empty($lng)) {
            return '';
        }

        // Convert to float to ensure proper formatting
        $latitude = (float) $lat;
        $longitude = (float) $lng;

        // Validate latitude and longitude ranges
        if ($latitude < -90 || $latitude > 90 || $longitude < -180 || $longitude > 180) {
            return '';
        }

        // Base Google Maps URL with coordinates
        $url = "https://www.google.com/maps?q={$latitude},{$longitude}";

        // Add label if provided
        if (!empty($label)) {
            $encodedLabel = urlencode($label);
            $url = "https://www.google.com/maps/place/{$encodedLabel}/@{$latitude},{$longitude},15z";
        }

        return $url;
    }
}

if (! function_exists('esp_get_profile_meta_count')) {
    function esp_get_profile_meta_count($key)
    {
        $users = \App\Models\User\User::whereHas('roles', function ($q) {
            $q->where('name', 'Service');
        });
        $count = 0;
        if ($key === 'emergency') {
            // Count members with emergency service enabled
            $count = $users->whereHas('profile', function ($q) {
                $q->whereExists(function ($query) {
                    $query->from('user_profiles_meta')
                        ->whereColumn('user_profiles_meta.user_profile_id', 'user_profiles.id')
                        ->where('user_profiles_meta.key', 'emergency_service_enabled')
                        ->where('user_profiles_meta.value', '1');
                });
            })->count();
        } elseif ($key === 'reviews') {
            // Count members with reviews and ratings
            $count = $users->whereHas('profile', function ($q) {
                $q->where('profile_rating', '>', 0);
            })->count();
        } elseif ($key === 'open_now') {
            // Count members that are currently open
            $count = $users->whereHas('profile', function ($q) {
                    $q->whereExists(function ($query) {
                        $query->from('user_profiles_meta')
                            ->whereColumn('user_profiles_meta.user_profile_id', 'user_profiles.id')
                            ->where('user_profiles_meta.key', 'business_hours')
                            ->where('user_profiles_meta.value', 'LIKE', '%"is_open":true%');
                    });
            })->count();
        }

        return $count;
    }
    
}

if (! function_exists('esp_mask_email')) {
    function esp_mask_email($email) {
        $emailLength = strlen($email);
        $maskedEmail = substr($email, 0, 3) . str_repeat('.', $emailLength - 9) . substr($email, -6);
        return $maskedEmail;
    }
}
