<?php

use App\Services\Settings\SettingsManager;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

if (! function_exists('get_setting')) {
    function get_setting($key, $default = null)
    {
        try {
            // Check if app is available and database connection exists
            if (! app()->bound(SettingsManager::class)) {
                return $default;
            }

            // Check if we're in console mode during early boot
            if (app()->runningInConsole() && ! app()->isBooted()) {
                return $default;
            }

            return app(SettingsManager::class)->get($key, $default);
        } catch (\Exception $e) {
            // Fall back to default if any error occurs (database not available, etc.)
            return $default;
        }
    }
}

if (! function_exists('get_setting_safe')) {
    function get_setting_safe($group, $key, $default = null)
    {
        try {
            // Check if app is available and database connection exists
            if (! app()->bound(SettingsManager::class)) {
                return $default;
            }

            // Check if we're in console mode during early boot
            if (app()->runningInConsole() && ! app()->isBooted()) {
                return $default;
            }

            return app(SettingsManager::class)->getByGroup($group, $key, $default);
        } catch (\Exception $e) {
            // Fall back to default if any error occurs
            return $default;
        }
    }
}

if (! function_exists('get_general_setting')) {
    function get_general_setting($key, $default = null)
    {
        return get_setting_safe('general', $key, $default);
    }
}

if (! function_exists('get_email_setting')) {
    function get_email_setting($key, $default = null)
    {
        return get_setting_safe('email', $key, $default);
    }
}

if (! function_exists('get_contact_setting')) {
    function get_contact_setting($key, $default = null)
    {
        return get_setting_safe('contact', $key, $default);
    }
}

if (! function_exists('get_site_name')) {
    function get_site_name()
    {
        return get_general_setting('app_name', config('app.name', 'Laravel'));
    }
}

if (! function_exists('get_site_tagline')) {
    function get_site_tagline()
    {
        return get_general_setting('app_tagline', '');
    }
}

if (! function_exists('get_admin_email')) {
    function get_admin_email()
    {
        return get_general_setting('admin_email', 'info@espmarketplace.com');
    }
}

if (! function_exists('get_app_currency')) {
    function get_app_currency()
    {
        return get_general_setting('app_currency', 'USD');
    }
}

if (! function_exists('get_app_timezone')) {
    function get_app_timezone()
    {
        return get_general_setting('app_timezone', config('app.timezone', 'UTC'));
    }
}

if (! function_exists('get_maintenance_mode')) {
    function get_maintenance_mode()
    {
        return get_general_setting('maintenance_mode', 'disabled') === 'enabled';
    }
}

if (! function_exists('get_ligth_logo')) {
    function get_ligth_logo()
    {
        $logo = '';
        try {
            $logo_file = get_general_setting('ligth_logo');
            if (! empty($logo_file) && function_exists('storage_path')) {
                $logo = Storage::url($logo_file);
            } else {
                $logo = asset('images/logo-white.svg');
            }

        } catch (\Exception $e) {
            // Fall back to default
            Log::error($e->getMessage());
        }
        return $logo;
    }
}

if (! function_exists('get_dark_logo')) {
    function get_dark_logo()
    {
        try {
            $logo_file = get_general_setting('dark_logo');
            if ($logo_file && function_exists('storage_path')) {
                return Storage::url($logo_file);
            }
        } catch (\Exception $e) {
            // Fall back to default
        }
        return asset('themes/ep/' . env('ACTIVE_THEME', 'esp_theme') . '/img/dark_logo.png');
    }
}

if (! function_exists('get_mobile_logo')) {
    function get_mobile_logo()
    {
        $logo = '';
        try {
            $logo_file = get_general_setting('ligth_logo');
            if ($logo_file && function_exists('storage_path')) {
                $logo = Storage::url($logo_file);
            }
            else{
                $logo = asset('images/logo-white.svg');
            }
        } catch (\Exception $e) {
            // Fall back to default
            Log::error($e->getMessage());
        }
        return $logo;
    }
}

if (! function_exists('get_app_favicon')) {
    function get_app_favicon()
    {
        try {
            $favicon_file = get_general_setting('app_favicon');
            if ($favicon_file && function_exists('storage_path')) {
                return Storage::url($favicon_file);
            }
        } catch (\Exception $e) {
            // Fall back to default
        }
        return asset('themes/ep/' . env('ACTIVE_THEME', 'esp_theme') . '/img/favicon.ico');
    }
}

// Email-specific helper functions
if (! function_exists('get_smtp_config')) {
    function get_smtp_config()
    {
        return [
            'host'       => get_email_setting('smtp_host', env('MAIL_HOST', '127.0.0.1')),
            'port'       => get_email_setting('smtp_port', env('MAIL_PORT', 587)),
            'username'   => get_email_setting('smtp_username', env('MAIL_USERNAME')),
            'password'   => get_email_setting('smtp_password', env('MAIL_PASSWORD')),
            'encryption' => get_email_setting('smtp_encryption', env('MAIL_ENCRYPTION', 'tls')),
        ];
    }
}

if (! function_exists('get_mail_from_config')) {
    function get_mail_from_config()
    {
        return [
            'address' => get_email_setting('from_address', env('MAIL_FROM_ADDRESS', 'hello@example.com')),
            'name'    => get_email_setting('from_name', env('MAIL_FROM_NAME', get_site_name())),
        ];
    }
}

// Payment-specific helper functions
if (! function_exists('get_payment_setting')) {
    function get_payment_setting($key, $default = null)
    {
        return get_setting_safe('payments', $key, $default);
    }
}

if (! function_exists('get_stripe_mode')) {
    function get_stripe_mode()
    {
        return get_payment_setting('stripe_mode', 'test');
    }
}

if (! function_exists('is_stripe_live_mode')) {
    function is_stripe_live_mode()
    {
        return get_stripe_mode() === 'live';
    }
}

if (! function_exists('get_stripe_publishable_key')) {
    function get_stripe_publishable_key()
    {
        if (is_stripe_live_mode()) {
            return get_payment_setting('stripe_publishable_key', env('STRIPE_KEY'));
        }

        return get_payment_setting('stripe_test_publishable_key', env('STRIPE_TEST_KEY'));
    }
}

if (! function_exists('get_stripe_secret_key')) {
    function get_stripe_secret_key()
    {
        if (is_stripe_live_mode()) {
            return get_payment_setting('stripe_secret_key', env('STRIPE_SECRET'));
        }

        return get_payment_setting('stripe_test_secret_key', env('STRIPE_TEST_SECRET'));
    }
}

if (! function_exists('get_stripe_webhook_secret')) {
    function get_stripe_webhook_secret()
    {
        return get_payment_setting('stripe_webhook_secret', env('STRIPE_WEBHOOK_SECRET'));
    }
}

if (! function_exists('get_stripe_currency')) {
    function get_stripe_currency()
    {
        return get_payment_setting('stripe_currency', env('CASHIER_CURRENCY', 'usd'));
    }
}

if (! function_exists('get_stripe_webhook_url')) {
    function get_stripe_webhook_url()
    {
        return get_payment_setting('stripe_webhook_url', '/webhook/stripe');
    }
}

if (! function_exists('is_stripe_enabled')) {
    function is_stripe_enabled()
    {
        return get_payment_setting('stripe_enabled', false);
    }
}

if (! function_exists('get_stripe_config')) {
    function get_stripe_config()
    {
        return [
            'mode'            => get_stripe_mode(),
            'publishable_key' => get_stripe_publishable_key(),
            'secret_key'      => get_stripe_secret_key(),
            'webhook_secret'  => get_stripe_webhook_secret(),
            'currency'        => get_stripe_currency(),
            'webhook_url'     => get_stripe_webhook_url(),
            'enabled'         => is_stripe_enabled(),
        ];
    }
}

// PayPal-specific helper functions
if (! function_exists('get_paypal_mode')) {
    function get_paypal_mode()
    {
        return get_payment_setting('paypal_mode', 'sandbox');
    }
}

if (! function_exists('is_paypal_live_mode')) {
    function is_paypal_live_mode()
    {
        return get_paypal_mode() === 'live';
    }
}

if (! function_exists('get_paypal_client_id')) {
    function get_paypal_client_id()
    {
        if (is_paypal_live_mode()) {
            return get_payment_setting('paypal_client_id', env('PAYPAL_CLIENT_ID'));
        }

        return get_payment_setting('paypal_sandbox_client_id', env('PAYPAL_SANDBOX_CLIENT_ID'));
    }
}

if (! function_exists('get_paypal_client_secret')) {
    function get_paypal_client_secret()
    {
        if (is_paypal_live_mode()) {
            return get_payment_setting('paypal_client_secret', env('PAYPAL_CLIENT_SECRET'));
        }

        return get_payment_setting('paypal_sandbox_client_secret', env('PAYPAL_SANDBOX_CLIENT_SECRET'));
    }
}

if (! function_exists('is_paypal_enabled')) {
    function is_paypal_enabled()
    {
        return get_payment_setting('paypal_enabled', false);
    }
}

if (! function_exists('get_paypal_config')) {
    function get_paypal_config()
    {
        return [
            'mode'          => get_paypal_mode(),
            'client_id'     => get_paypal_client_id(),
            'client_secret' => get_paypal_client_secret(),
            'currency'      => get_payment_setting('paypal_currency', 'USD'),
            'webhook_url'   => get_payment_setting('paypal_webhook_url', '/webhook/paypal'),
            'enabled'       => is_paypal_enabled(),
        ];
    }
}

// General payment helper functions
if (! function_exists('get_default_payment_gateway')) {
    function get_default_payment_gateway()
    {
        return get_payment_setting('default_payment_gateway', 'stripe');
    }
}

if (! function_exists('get_payment_tax_rate')) {
    function get_payment_tax_rate()
    {
        return get_payment_setting('payment_tax_rate', 0);
    }
}

if (! function_exists('get_payment_processing_fee')) {
    function get_payment_processing_fee()
    {
        return get_payment_setting('payment_processing_fee', 0);
    }
}

if (! function_exists('get_payment_minimum_amount')) {
    function get_payment_minimum_amount()
    {
        return get_payment_setting('payment_minimum_amount', 1.00);
    }
}

// Credit system helper functions
if (! function_exists('get_membership_setting')) {
    function get_membership_setting($key, $default = null)
    {
        return get_setting_safe('membership', $key, $default);
    }
}

if (! function_exists('get_credit_price')) {
    function get_credit_price()
    {
        return get_membership_setting('credit_price', 1.00);
    }
}

if (! function_exists('get_credit_currency')) {
    function get_credit_currency()
    {
        return get_membership_setting('credit_currency', 'USD');
    }
}

if (! function_exists('is_bulk_discounts_enabled')) {
    function is_bulk_discounts_enabled()
    {
        return get_membership_setting('bulk_discounts_enabled', true);
    }
}

if (! function_exists('get_credit_bulk_discounts')) {
    function get_credit_bulk_discounts()
    {
        $discounts = get_membership_setting('credit_bulk_discounts', json_encode([
            ['quantity' => 10, 'discount' => 5],
            ['quantity' => 25, 'discount' => 10],
            ['quantity' => 50, 'discount' => 15],
            ['quantity' => 100, 'discount' => 20],
        ]));

        return is_string($discounts) ? json_decode($discounts, true) : $discounts;
    }
}

if (! function_exists('get_max_credits_per_purchase')) {
    function get_max_credits_per_purchase()
    {
        return get_membership_setting('max_credits_per_purchase', 1000);
    }
}

if (! function_exists('get_min_credits_per_purchase')) {
    function get_min_credits_per_purchase()
    {
        return get_membership_setting('min_credits_per_purchase', 1);
    }
}

if (! function_exists('is_credit_expiry_enabled')) {
    function is_credit_expiry_enabled()
    {
        return get_membership_setting('credit_expiry_enabled', false);
    }
}

if (! function_exists('get_credit_expiry_days')) {
    function get_credit_expiry_days()
    {
        return get_membership_setting('credit_expiry_days', 365);
    }
}

if (! function_exists('is_credit_refund_enabled')) {
    function is_credit_refund_enabled()
    {
        return get_membership_setting('credit_refund_enabled', false);
    }
}

if (! function_exists('is_credit_transfer_enabled')) {
    function is_credit_transfer_enabled()
    {
        return get_membership_setting('credit_transfer_enabled', false);
    }
}

if (! function_exists('get_footer_settings')) {
    function get_footer_settings($key, $default = null)
    {
        return get_setting_safe('footer', $key, $default);
    }
}

if (! function_exists('get_footer_light_logo')) {
    function get_footer_light_logo()
    {
        $footer_logo = '';
        try {
            $logo_file = get_footer_settings('light_logo');
            if ($logo_file && function_exists('storage_path')) {
                $footer_logo = Storage::url($logo_file);
            }else{
                $footer_logo = asset('images/logo-white.svg');
            }
        } catch (\Exception $e) {
            // Fall back to default
            
        }
        return $footer_logo;
    }
}

if (! function_exists('get_footer_dark_logo')) {
    function get_footer_dark_logo()
    {
        try {
            $logo_file = get_footer_settings('dark_logo');
            if ($logo_file && function_exists('storage_path')) {
                return Storage::url($logo_file);
            }
        } catch (\Exception $e) {
            // Fall back to default
        }
        return asset('themes/ep/' . env('ACTIVE_THEME', 'esp_theme') . '/img/dark_logo.png');
    }
}

if (! function_exists('get_app_about')) {
    function get_app_about()
    {
        $about = get_footer_settings(key: 'footer_app_about', default: 'Welcome to our app!');
        return $about;
    }
}

if (! function_exists('get_footer_social_links')) {
    function get_footer_social_links()
    {
        $facebook      = get_footer_settings('facebook');
        $twitter       = get_footer_settings('twitter');
        $linkedin      = get_footer_settings('linkedin');
        $instagram     = get_footer_settings('instagram');
        $socical_links = [
            'facebook'  => $facebook,
            'twitter'   => $twitter,
            'linkedin'  => $linkedin,
            'instagram' => $instagram,
        ];
        return $socical_links;
    }
}
if (! function_exists('get_footer_copyright')) {
    function get_footer_copyright()
    {
        $copyright = get_footer_settings(key: 'copyright_text', default: '© ' . date('Y') . ' ESP Marketplace. All Rights Reserved.');
        return $copyright;
    }
}

if (! function_exists('get_footer_menu_1')) {
    function get_footer_menu_1()
    {
        $enable_menu = get_footer_settings(key: 'enable_footer_menu_1', default: '0');
        if ($enable_menu == '1') {
            $menu = get_footer_settings(key: 'choose_menu_1');
            return $menu;
        }
        return null;
    }
}
if (! function_exists('get_footer_menu_2')) {
    function get_footer_menu_2()
    {
        $enable_menu = get_footer_settings(key: 'enable_footer_menu_2', default: '0');
        if ($enable_menu == '1') {
            $menu = get_footer_settings(key: 'choose_menu_2');

            return $menu;
        }
        return null;
    }
}

if (! function_exists('get_footer_subscription')) {
    function get_footer_subscription()
    {
        $enable_subscription = get_footer_settings(key: 'enable_footer_subscription', default: '0');
        $subscription        = [];
        if ($enable_subscription == '1') {
            $heading      = get_footer_settings(key: 'footer_subscription_heading', default: 'Subscribe to our newsletter');
            $sub_heading  = get_footer_settings(key: 'footer_subscription_sub_heading', default: 'Get new job opportunities delivered directly to your inbox!');
            $helper_text  = get_footer_settings(key: 'footer_subscription_helper_text', default: 'We respect your privacy. Unsubscribe anytime.');
            $subscription = [
                'heading'     => $heading,
                'sub_heading' => $sub_heading,
                'helper_text' => $helper_text,
            ];
            return $subscription;
        }
        return null;
    }
}
