<?php

namespace App\Http\Controllers\Api\V1\Auth;

use App\Http\Controllers\Api\ApiController;
use App\Models\User\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class AuthController extends ApiController
{
    /**
     * Login user and create token
     */
    public function login(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'email' => 'required|email',
                'password' => 'required',
                'device_name' => 'string|max:255',
            ]);

            $user = User::where('email', $request->email)->first();

            if (!$user || !Hash::check($request->password, $user->password)) {
                throw ValidationException::withMessages([
                    'email' => ['The provided credentials are incorrect.'],
                ]);
            }

            // Check if user is active
            if ($user->deleted_at) {
                return $this->errorResponse('Your account has been deactivated.', 403);
            }

            // Create token with abilities/permissions
            $deviceName = $request->device_name ?? $request->userAgent();
            $token = $user->createToken($deviceName)->plainTextToken;

            return $this->successResponse([
                'token' => $token,
                'token_type' => 'Bearer',
                'user' => [
                    'id' => $user->uuid,
                    'name' => $user->name,
                    'email' => $user->email,
                    'roles' => $user->roles->pluck('name'),
                ],
            ], 'Login successful');

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Login failed: ' . $e->getMessage());
        }
    }

    /**
     * Register a new user
     */
    public function register(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:users',
                'password' => 'required|string|min:8|confirmed',
                'device_name' => 'string|max:255',
            ]);

            $user = User::create([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'password' => Hash::make($validated['password']),
            ]);

            // Create token
            $deviceName = $request->device_name ?? $request->userAgent();
            $token = $user->createToken($deviceName)->plainTextToken;

            return $this->createdResponse([
                'token' => $token,
                'token_type' => 'Bearer',
                'user' => [
                    'id' => $user->uuid,
                    'name' => $user->name,
                    'email' => $user->email,
                ],
            ], 'Registration successful');

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Registration failed: ' . $e->getMessage());
        }
    }

    /**
     * Logout user (Revoke current token)
     */
    public function logout(Request $request): JsonResponse
    {
        try {
            // Revoke current user token
            $request->user()->currentAccessToken()->delete();

            return $this->successResponse(null, 'Logged out successfully');

        } catch (\Exception $e) {
            return $this->serverErrorResponse('Logout failed: ' . $e->getMessage());
        }
    }

    /**
     * Logout from all devices (Revoke all tokens)
     */
    public function logoutAll(Request $request): JsonResponse
    {
        try {
            // Revoke all user tokens
            $request->user()->tokens()->delete();

            return $this->successResponse(null, 'Logged out from all devices successfully');

        } catch (\Exception $e) {
            return $this->serverErrorResponse('Logout failed: ' . $e->getMessage());
        }
    }

    /**
     * Get authenticated user details
     */
    public function me(Request $request): JsonResponse
    {
        try {
            $user = $request->user();

            return $this->successResponse([
                'id' => $user->uuid,
                'name' => $user->name,
                'email' => $user->email,
                'email_verified_at' => $user->email_verified_at?->toIso8601String(),
                'roles' => $user->roles->pluck('name'),
                'permissions' => $user->getAllPermissions()->pluck('name'),
            ], 'User details retrieved successfully');

        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to retrieve user: ' . $e->getMessage());
        }
    }

    /**
     * Refresh token (create new token and revoke old one)
     */
    public function refresh(Request $request): JsonResponse
    {
        try {
            $user = $request->user();
            $deviceName = $request->device_name ?? $request->userAgent();

            // Revoke current token
            $request->user()->currentAccessToken()->delete();

            // Create new token
            $token = $user->createToken($deviceName)->plainTextToken;

            return $this->successResponse([
                'token' => $token,
                'token_type' => 'Bearer',
            ], 'Token refreshed successfully');

        } catch (\Exception $e) {
            return $this->serverErrorResponse('Token refresh failed: ' . $e->getMessage());
        }
    }

    /**
     * Verify Sanctum access token
     */
    public function verifyToken(Request $request): JsonResponse
    {
        try {
            // Extract token from Authorization header
            $authHeader = $request->header('Authorization');

            if (!$authHeader || !str_starts_with($authHeader, 'Bearer ')) {
                return $this->errorResponse('Token is invalid', 401, null, 'INVALID_TOKEN');
            }

            // Remove "Bearer " prefix and get plain token
            $plainToken = substr($authHeader, 7);

            if (empty($plainToken)) {
                return $this->errorResponse('Token is invalid', 401, null, 'INVALID_TOKEN');
            }

            // Sanctum tokens have format: {id}|{plainToken}
            // We need to extract the part after the pipe and hash it
            if (str_contains($plainToken, '|')) {
                $tokenParts = explode('|', $plainToken, 2);
                if (count($tokenParts) !== 2) {
                    return $this->errorResponse('Token is invalid', 401, null, 'INVALID_TOKEN');
                }
                $plainToken = $tokenParts[1]; // Get the actual token part after the pipe
            }

            // Hash the token (Sanctum stores hashed tokens)
            $hashedToken = hash('sha256', $plainToken);

            // Look up token in personal_access_tokens table
            $accessToken = \Laravel\Sanctum\PersonalAccessToken::where('token', $hashedToken)->first();

            if (!$accessToken) {
                return $this->errorResponse('Token is invalid', 401, null, 'INVALID_TOKEN');
            }

            // Check if token has expired
            if ($accessToken->expires_at && $accessToken->expires_at->isPast()) {
                return $this->errorResponse('Token has expired', 401, [
                    'status' => 'expired',
                    'expired_at' => $accessToken->expires_at->toIso8601String(),
                ], 'TOKEN_EXPIRED');
            }

            // Load the tokenable (should be ApiClient)
            $tokenable = $accessToken->tokenable;

            // Verify it's an ApiClient
            if (!$tokenable instanceof \App\Models\ApiClient) {
                return $this->errorResponse('Token is invalid', 401, null, 'INVALID_TOKEN');
            }

            // Check if API client is active
            if (!$tokenable->is_active) {
                return $this->errorResponse('API client is inactive', 401, [
                    'status' => 'invalid',
                    'reason' => 'api_client_inactive',
                ], 'API_CLIENT_INACTIVE');
            }

            // Token is valid - return success with API client details
            return $this->successResponse([
                'status' => 'valid',
                'api_client' => [
                    'id' => $tokenable->id,
                    'name' => $tokenable->name,
                    'type' => $tokenable->type->value,
                    'is_active' => $tokenable->is_active,
                ],
                'token_info' => [
                    'name' => $accessToken->name,
                    'created_at' => $accessToken->created_at->toIso8601String(),
                    'last_used_at' => $accessToken->last_used_at?->toIso8601String(),
                    'expires_at' => $accessToken->expires_at?->toIso8601String(),
                ],
            ], 'Token is valid');

        } catch (\Exception $e) {
            return $this->serverErrorResponse('Token verification failed: ' . $e->getMessage());
        }
    }
}
