<?php

namespace App\Http\Controllers\Api\V1\Lead;

use App\Http\Controllers\Api\ApiController;
use App\Http\Resources\V1\Lead\Lead\LeadDetailResource;
use App\Services\Lead\LeadService;
use App\Enums\LeadStatus;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class LeadAssignmentController extends ApiController
{
    protected LeadService $leadService;

    public function __construct(LeadService $leadService)
    {
        $this->leadService = $leadService;
    }

    /**
     * Assign lead to users
     */
    public function assign(Request $request, string $uuid): JsonResponse
    {
        try {
            $validated = $request->validate([
                'assigned_user_ids' => 'required|array',
                'assigned_user_ids.*' => 'integer|exists:users,id',
                'user_statuses' => 'nullable|array',
                'user_statuses.*' => \Illuminate\Validation\Rule::in(array_column(LeadStatus::cases(), 'value')),
                'sync_users' => 'sometimes|boolean',
            ]);

            $lead = $this->leadService->assignToUsers(
                $uuid,
                $validated['assigned_user_ids'],
                $validated['user_statuses'] ?? [],
                $validated['sync_users'] ?? true
            );

            return $this->successResponse(
                new LeadDetailResource($lead),
                'Lead assigned successfully'
            );

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to assign lead: ' . $e->getMessage());
        }
    }

    /**
     * Update lead status
     */
    public function updateStatus(Request $request, string $uuid): JsonResponse
    {
        try {
            $validated = $request->validate([
                'lead_status' => ['required', 'string', \Illuminate\Validation\Rule::in(array_column(LeadStatus::cases(), 'value'))],
                'user_id' => ['nullable', 'integer', 'exists:users,id'],
            ]);

            $lead = $this->leadService->updateStatus(
                $uuid,
                $validated['lead_status'],
                $validated['user_id'] ?? null
            );

            return $this->successResponse(
                new LeadDetailResource($lead),
                'Lead status updated successfully'
            );

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to update status: ' . $e->getMessage());
        }
    }
}
