<?php

namespace App\Http\Controllers\Api\V1\Lead;

use App\DTOs\Lead\Lead\CreateLeadDTO;
use App\DTOs\Lead\Lead\LeadFilterDTO;
use App\DTOs\Lead\Lead\UpdateLeadDTO;
use App\Http\Controllers\Api\ApiController;
use App\Http\Resources\V1\Lead\Lead\LeadDetailResource;
use App\Http\Resources\V1\Lead\Lead\LeadResource;
use App\Services\Lead\LeadAddressService;
use App\Services\Lead\LeadFormService;
use App\Services\Lead\LeadService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class LeadController extends ApiController
{
    protected LeadService $leadService;

    protected LeadFormService $leadFormService;

    protected LeadAddressService $leadAddressService;

    public function __construct(
        LeadService $leadService,
        LeadFormService $leadFormService,
        LeadAddressService $leadAddressService
    ) {
        $this->leadService = $leadService;
        $this->leadFormService = $leadFormService;
        $this->leadAddressService = $leadAddressService;
    }

    /**
     * Display a listing of leads
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $filterDTO = LeadFilterDTO::fromRequest($request);

            $leads = $this->leadService->getFiltered($filterDTO);

            return $this->successResponse(
                LeadResource::collection($leads),
                'Leads retrieved successfully'
            );

        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to retrieve leads: '.$e->getMessage());
        }
    }

    /**
     * Store a newly created lead
     */
    public function store(Request $request): JsonResponse
    {
        try {
            // Get the lead form by UUID and resolve to database ID
            if ($request->has('form_id')) {
                $leadForm = $this->leadFormService->findByUuidOrFail($request->input('form_id'));
                $request->merge(['lead_form_id' => $leadForm->id]);
            }

            // Decode lead_data if it's a JSON string
            if ($request->has('lead_data') && ! empty($request->input('lead_data'))) {
                $leadData = is_string($request->input('lead_data')) ? json_decode($request->input('lead_data'), true) : $request->input('lead_data');
                if (json_last_error() === JSON_ERROR_NONE) {
                    $request->merge(['meta' => $leadData]);
                }
            }

            $request->merge(['lead_data' => []]);
            $createDTO = CreateLeadDTO::fromRequest($request);
            $createDTO->created_by = auth()->id();

            $lead = $this->leadService->create($createDTO);

            // Create lead address if location data is provided
            $locationData = [
                'address' => $request->input('address'),
                'lat' => $request->input('lat'),
                'lng' => $request->input('lng'),
                'city' => $request->input('city'),
                'state' => $request->input('state'),
                'state_code' => $request->input('state_code'),
                'country' => $request->input('country'),
                'country_code' => $request->input('country_code'),
            ];

            $this->leadAddressService->createFromLocationData($lead->id, $locationData);

            // Handle file attachments from base64 data
            if ($request->has('attachments') && is_array($request->input('attachments'))) {
                $this->processAttachments($lead, $request->input('attachments'));
            }

            return $this->createdResponse(
                new LeadDetailResource($lead),
                'Lead created successfully'
            );

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to create lead: '.$e->getMessage());
        }
    }

    /**
     * Display the specified lead
     */
    public function show(string $uuid): JsonResponse
    {
        try {
            $lead = $this->leadService->findByUuidOrFail($uuid, ['leadForm.formFields', 'users', 'author']);

            return $this->successResponse(
                new LeadDetailResource($lead),
                'Lead retrieved successfully'
            );

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to retrieve lead: '.$e->getMessage());
        }
    }

    /**
     * Update the specified lead
     */
    public function update(Request $request, string $uuid): JsonResponse
    {
        try {
            $updateDTO = UpdateLeadDTO::fromRequest($request);

            $lead = $this->leadService->update($uuid, $updateDTO);

            return $this->successResponse(
                new LeadDetailResource($lead),
                'Lead updated successfully'
            );

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to update lead: '.$e->getMessage());
        }
    }

    /**
     * Remove the specified lead
     */
    public function destroy(string $uuid): JsonResponse
    {
        try {
            $this->leadService->delete($uuid);

            return $this->noContentResponse('Lead deleted successfully');

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to delete lead: '.$e->getMessage());
        }
    }

    /**
     * Process and attach files from base64 data
     *
     * @param  \App\Models\Lead\Lead  $lead
     * @param  array  $attachments  Array of attachment data with base64 encoded files
     */
    protected function processAttachments($lead, array $attachments): void
    {
        foreach ($attachments as $attachment) {
            // Skip if required data is missing
            if (empty($attachment['data']) || empty($attachment['name'])) {
                continue;
            }

            try {
                // Extract base64 data
                $base64Data = $attachment['data'];

                // Check if it's a data URL (e.g., "data:image/png;base64,...")
                if (preg_match('/^data:([^;]+);base64,(.+)$/', $base64Data, $matches)) {
                    $mimeType = $matches[1];
                    $base64Content = $matches[2];
                } else {
                    // Assume it's just base64 without data URL prefix
                    $base64Content = $base64Data;
                    $mimeType = $attachment['type'] ?? 'application/octet-stream';
                }

                // Decode base64 content
                $fileContent = base64_decode($base64Content);

                if ($fileContent === false) {
                    // Skip invalid base64 data
                    continue;
                }

                // Validate file size (optional - add limits as needed)
                $fileSizeKB = strlen($fileContent) / 1024;
                $maxSizeKB = $attachment['max_size_kb'] ?? 10240; // 10MB default

                if ($fileSizeKB > $maxSizeKB) {
                    // Skip files that exceed size limit
                    continue;
                }

                // Add file to media library
                $media = $lead->addMediaFromString($fileContent)
                    ->usingName(pathinfo($attachment['name'], PATHINFO_FILENAME))
                    ->usingFileName($this->sanitizeFileName($attachment['name']))
                    ->withCustomProperties([
                        'field_id' => $attachment['field_id'] ?? null,
                        'field_name' => $attachment['field_name'] ?? null,
                        'original_size' => $attachment['size'] ?? strlen($fileContent),
                    ])
                    ->toMediaCollection('attachments');

            } catch (\Exception $e) {
                // Log error but continue processing other files
                Log::error('Failed to process attachment: '.$e->getMessage(), [
                    'attachment' => $attachment['name'] ?? 'unknown',
                    'lead_id' => $lead->id,
                ]);

                continue;
            }
        }
    }

    /**
     * Sanitize filename to prevent security issues
     */
    protected function sanitizeFileName(string $filename): string
    {
        // Get file extension
        $extension = pathinfo($filename, PATHINFO_EXTENSION);
        $basename = pathinfo($filename, PATHINFO_FILENAME);

        // Remove any special characters and spaces
        $basename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $basename);

        // Limit length
        $basename = substr($basename, 0, 200);

        // Add timestamp to ensure uniqueness
        $basename = $basename.'_'.time();

        return $extension ? $basename.'.'.$extension : $basename;
    }
}
