<?php

namespace App\Http\Controllers\Api\V1\Lead;

use App\Http\Controllers\Api\ApiController;
use App\Http\Resources\V1\Lead\LeadForm\LeadFormResource;
use App\Http\Resources\V1\Lead\LeadForm\LeadFormDetailResource;
use App\DTOs\Lead\LeadForm\CreateLeadFormDTO;
use App\DTOs\Lead\LeadForm\UpdateLeadFormDTO;
use App\DTOs\Lead\LeadForm\LeadFormFilterDTO;
use App\Services\Lead\LeadFormService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class LeadFormController extends ApiController
{
    protected LeadFormService $leadFormService;

    public function __construct(LeadFormService $leadFormService)
    {
        $this->leadFormService = $leadFormService;
    }

    /**
     * Display a listing of lead forms
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $filterDTO = LeadFormFilterDTO::fromRequest($request);

            $leadForms = $this->leadFormService->getFiltered($filterDTO);

            return $this->successResponse(
                LeadFormResource::collection($leadForms),
                'Lead forms retrieved successfully'
            );

        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to retrieve lead forms: ' . $e->getMessage());
        }
    }

    /**
     * Store a newly created lead form
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $createDTO = CreateLeadFormDTO::fromRequest($request);
            $createDTO->created_by = auth()->id();

            $leadForm = $this->leadFormService->create($createDTO);

            return $this->createdResponse(
                new LeadFormDetailResource($leadForm),
                'Lead form created successfully'
            );

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to create lead form: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified lead form
     */
    public function show(string $uuid): JsonResponse
    {
        try {
            $leadForm = $this->leadFormService->findByUuidOrFail(
                $uuid,
                ['formFields', 'memberCategories'],
                ['visits', 'leads']
            );

            return $this->successResponse(
                new LeadFormDetailResource($leadForm),
                'Lead form retrieved successfully'
            );

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to retrieve lead form: ' . $e->getMessage());
        }
    }

    /**
     * Update the specified lead form
     */
    public function update(Request $request, string $uuid): JsonResponse
    {
        try {
            $updateDTO = UpdateLeadFormDTO::fromRequest($request);

            $leadForm = $this->leadFormService->update($uuid, $updateDTO);

            return $this->successResponse(
                new LeadFormDetailResource($leadForm),
                'Lead form updated successfully'
            );

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to update lead form: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified lead form
     */
    public function destroy(string $uuid): JsonResponse
    {
        try {
            $this->leadFormService->delete($uuid);

            return $this->noContentResponse('Lead form deleted successfully');

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to delete lead form: ' . $e->getMessage());
        }
    }

    /**
     * Get lead form statistics
     */
    public function stats(string $uuid): JsonResponse
    {
        try {
            $stats = $this->leadFormService->getStats($uuid);

            return $this->successResponse($stats, 'Lead form statistics retrieved successfully');

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to retrieve statistics: ' . $e->getMessage());
        }
    }
}
