<?php

namespace App\Http\Controllers\Api\V1\Lead;

use App\Http\Controllers\Api\ApiController;
use App\Http\Resources\V1\Lead\LeadForm\LeadFormFieldResource;
use App\Services\Lead\LeadFormFieldService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class LeadFormFieldController extends ApiController
{
    protected LeadFormFieldService $leadFormFieldService;

    public function __construct(LeadFormFieldService $leadFormFieldService)
    {
        $this->leadFormFieldService = $leadFormFieldService;
    }

    /**
     * Get form fields with pivot data
     */
    public function index(string $uuid): JsonResponse
    {
        try {
            $formFields = $this->leadFormFieldService->getFormFields($uuid);

            return $this->successResponse(
                LeadFormFieldResource::collection($formFields),
                'Form fields retrieved successfully'
            );

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to retrieve fields: ' . $e->getMessage());
        }
    }

    /**
     * Attach field to form
     */
    public function store(Request $request, string $uuid): JsonResponse
    {
        try {
            $data = $request->validate([
                'field_id' => 'required|integer|exists:lead_form_fields,id',
                'field_placeholder' => 'nullable|string|max:255',
                'toggle_yes_text' => 'nullable|string|max:50',
                'toggle_no_text' => 'nullable|string|max:50',
                'field_columns' => 'nullable|integer|min:1|max:12',
                'help_text' => 'nullable|string|max:500',
                'field_options' => 'nullable|array',
                'field_options.*' => 'string|max:255',
                'is_required' => 'sometimes|boolean',
                'is_visible' => 'sometimes|boolean',
                'field_order' => 'sometimes|integer|min:0',
            ]);

            $fieldId = $data['field_id'];
            unset($data['field_id']);

            $formFields = $this->leadFormFieldService->attachField($uuid, $fieldId, $data);

            return $this->createdResponse(
                LeadFormFieldResource::collection($formFields),
                'Field attached successfully'
            );

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to attach field: ' . $e->getMessage());
        }
    }

    /**
     * Update field configuration
     */
    public function update(Request $request, string $uuid, int $fieldId): JsonResponse
    {
        try {
            $data = $request->validate([
                'field_placeholder' => ['nullable', 'string', 'max:255'],
                'toggle_yes_text' => ['nullable', 'string', 'max:50'],
                'toggle_no_text' => ['nullable', 'string', 'max:50'],
                'field_columns' => ['nullable', 'integer', 'min:1', 'max:12'],
                'help_text' => ['nullable', 'string', 'max:500'],
                'field_options' => ['nullable', 'array'],
                'is_required' => ['nullable', 'boolean'],
                'is_visible' => ['nullable', 'boolean'],
                'field_order' => ['nullable', 'integer', 'min:0'],
            ]);

            $formFields = $this->leadFormFieldService->updateFieldConfiguration($uuid, $fieldId, $data);

            return $this->successResponse(
                LeadFormFieldResource::collection($formFields),
                'Field configuration updated successfully'
            );

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to update field: ' . $e->getMessage());
        }
    }

    /**
     * Detach field from form
     */
    public function destroy(string $uuid, int $fieldId): JsonResponse
    {
        try {
            $this->leadFormFieldService->detachField($uuid, $fieldId);

            return $this->noContentResponse('Field detached successfully');

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to detach field: ' . $e->getMessage());
        }
    }
}
