<?php

namespace App\Http\Controllers\Api\V1\Lead;

use App\Http\Controllers\Api\ApiController;
use App\Http\Resources\V1\Lead\LeadFormVisit\LeadFormVisitResource;
use App\Http\Resources\V1\Lead\LeadFormVisit\LeadFormVisitStatsResource;
use App\DTOs\Lead\LeadFormVisit\LeadFormVisitFilterDTO;
use App\DTOs\Lead\LeadFormVisit\CreateLeadFormVisitDTO;
use App\DTOs\Lead\LeadFormVisit\UpdateLeadFormVisitDTO;
use App\Services\Lead\LeadFormVisitService;
use App\Services\Lead\LeadFormService;
use App\Models\Lead\LeadForm;
use App\Models\Lead\LeadFormVisit;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class LeadFormVisitController extends ApiController
{
    protected LeadFormVisitService $visitService;
    protected LeadFormService $leadFormService;

    public function __construct(
        LeadFormVisitService $visitService,
        LeadFormService $leadFormService
    ) {
        $this->visitService = $visitService;
        $this->leadFormService = $leadFormService;
    }
    /**
     * Get form visits
     */
    public function index(Request $request, string $uuid): JsonResponse
    {
        try {
            $leadForm = LeadForm::where('uuid', $uuid)->firstOrFail();

            $filterDTO = LeadFormVisitFilterDTO::fromRequest($request);

            $query = LeadFormVisit::where('lead_form_id', $leadForm->id);

            // Apply filters
            if ($filterDTO->source) {
                $query->where('source', $filterDTO->source);
            }

            if ($filterDTO->country) {
                $query->where('country', $filterDTO->country);
            }

            if ($filterDTO->state) {
                $query->where('state', $filterDTO->state);
            }

            if ($filterDTO->city) {
                $query->where('city', $filterDTO->city);
            }

            // Date filters
            if ($filterDTO->visit_date_from) {
                $query->whereDate('visit_date', '>=', $filterDTO->visit_date_from);
            }
            if ($filterDTO->visit_date_to) {
                $query->whereDate('visit_date', '<=', $filterDTO->visit_date_to);
            }

            // Grouping for analytics
            if ($filterDTO->shouldGroup()) {
                return $this->getGroupedVisits($query, $filterDTO->group_by);
            }

            // Load relationships
            if ($filterDTO->with_lead_form) {
                $query->with('leadForm');
            }

            // Apply sorting
            $query->orderBy($filterDTO->order_by, $filterDTO->order_direction);

            // Paginate
            $visits = $query->paginate($filterDTO->per_page);

            return $this->successResponse(
                LeadFormVisitResource::collection($visits),
                'Visits retrieved successfully'
            );

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to retrieve visits: ' . $e->getMessage());
        }
    }

    /**
     * Get grouped visit statistics
     */
    private function getGroupedVisits($query, string $groupBy): JsonResponse
    {
        $stats = $query->select($groupBy, DB::raw('count(*) as count'))
            ->groupBy($groupBy)
            ->get()
            ->pluck('count', $groupBy)
            ->toArray();

        return $this->successResponse([
            'group_by' => $groupBy,
            'data' => $stats,
        ], 'Grouped visits retrieved successfully');
    }

    /**
     * Store a newly created visit
     */
    public function store(Request $request, string $uuid): JsonResponse
    {
        try {
            // Get the lead form by UUID
            $leadForm = $this->leadFormService->findByUuidOrFail($uuid);

            // Merge lead_form_id into request
            $request->merge(['lead_form_id' => $leadForm->id]);

            $createDTO = CreateLeadFormVisitDTO::fromRequest($request);

            $visit = $this->visitService->trackVisit($leadForm, $createDTO);

            return $this->createdResponse(
                new LeadFormVisitResource($visit),
                'Visit created successfully'
            );

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to create visit: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified visit
     */
    public function show(string $uuid, string $visitUuid): JsonResponse
    {
        try {
            // Verify the lead form exists
            $this->leadFormService->findByUuidOrFail($uuid);

            // Get the visit
            $visit = $this->visitService->findByUuidOrFail($visitUuid, ['leadForm']);

            // Verify the visit belongs to this lead form
            if ($visit->leadForm->uuid !== $uuid) {
                return $this->notFoundResponse('Visit not found for this lead form');
            }

            return $this->successResponse(
                new LeadFormVisitResource($visit),
                'Visit retrieved successfully'
            );

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form or visit not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to retrieve visit: ' . $e->getMessage());
        }
    }

    /**
     * Update the specified visit
     */
    public function update(Request $request, string $uuid, string $visitUuid): JsonResponse
    {
        try {
            // Verify the lead form exists
            $this->leadFormService->findByUuidOrFail($uuid);

            // Get the visit
            $visit = $this->visitService->findByUuidOrFail($visitUuid, ['leadForm']);

            // Verify the visit belongs to this lead form
            if ($visit->leadForm->uuid !== $uuid) {
                return $this->notFoundResponse('Visit not found for this lead form');
            }

            $updateDTO = UpdateLeadFormVisitDTO::fromRequest($request);

            $visit = $this->visitService->update($visitUuid, $updateDTO);

            return $this->successResponse(
                new LeadFormVisitResource($visit),
                'Visit updated successfully'
            );

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form or visit not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to update visit: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified visit
     */
    public function destroy(string $uuid, string $visitUuid): JsonResponse
    {
        try {
            // Verify the lead form exists
            $this->leadFormService->findByUuidOrFail($uuid);

            // Get the visit
            $visit = $this->visitService->findByUuidOrFail($visitUuid, ['leadForm']);

            // Verify the visit belongs to this lead form
            if ($visit->leadForm->uuid !== $uuid) {
                return $this->notFoundResponse('Visit not found for this lead form');
            }

            $this->visitService->delete($visitUuid);

            return $this->noContentResponse('Visit deleted successfully');

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form or visit not found');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to delete visit: ' . $e->getMessage());
        }
    }
}
