<?php

namespace App\Http\Controllers\Api\V1\Lead;

use App\Http\Controllers\Api\ApiController;
use App\Http\Resources\V1\Lead\Lead\LeadDetailResource;
use App\DTOs\Lead\Lead\CreateLeadDTO;
use App\Services\Lead\LeadService;
use App\Services\Lead\LeadFormService;
use App\Services\Lead\LeadFormVisitService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class LeadSubmissionController extends ApiController
{
    protected LeadService $leadService;
    protected LeadFormService $leadFormService;
    protected LeadFormVisitService $leadFormVisitService;

    public function __construct(
        LeadService $leadService,
        LeadFormService $leadFormService,
        LeadFormVisitService $leadFormVisitService
    ) {
        $this->leadService = $leadService;
        $this->leadFormService = $leadFormService;
        $this->leadFormVisitService = $leadFormVisitService;
    }

    /**
     * Submit a lead to a form (Public endpoint)
     */
    public function submit(Request $request, string $uuid): JsonResponse
    {
        try {
            $leadForm = $this->leadFormService->findActiveByUuidOrFail($uuid);

            // Create lead DTO with validation
            $createDTO = CreateLeadDTO::fromRequest($request);
            $createDTO->lead_form_id = $leadForm->id;

            // Set lead source from query params if not provided
            if (!$createDTO->lead_source) {
                $createDTO->lead_source = $request->query('utm_source', 'web');
            }

            // Create lead
            $lead = $this->leadService->create($createDTO);

            // Track visit automatically
            $this->leadFormVisitService->autoTrackVisit($request, $leadForm);

            return $this->createdResponse(
                new LeadDetailResource($lead),
                $leadForm->submit_message ?? 'Thank you for your submission!'
            );

        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found or is inactive');
        } catch (\Exception $e) {
            return $this->serverErrorResponse('Failed to submit lead: ' . $e->getMessage());
        }
    }

    /**
     * Track form visit (Public endpoint)
     */
    public function trackVisit(Request $request, string $uuid): JsonResponse
    {
        try {
            $leadForm = $this->leadFormService->findActiveByUuidOrFail($uuid);

            $this->leadFormVisitService->autoTrackVisit($request, $leadForm);

            return $this->successResponse(null, 'Visit tracked successfully');

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return $this->notFoundResponse('Lead form not found');
        } catch (\Exception $e) {
            // Silent fail for visit tracking
            return $this->successResponse(null, 'Visit tracking noted');
        }
    }
}
