<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Auth\Events\Verified;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\RedirectResponse;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Access\AuthorizationException;
use App\Enums\UserProfileStatus;
use App\Models\User\User;

class EmailVerificationController extends Controller
{
    public function __invoke(string $id, string $hash): RedirectResponse
    {
        // Find the user by ID instead of relying on Auth::user()
        $user = User::findOrFail($id);

        if (!hash_equals((string) $hash, sha1($user->getEmailForVerification()))) {
            throw new AuthorizationException();
        }

        if ($user->hasVerifiedEmail()) {
            // If user is already verified but not logged in, log them in
            if (!Auth::check()) {
                Auth::login($user, true);
            }
            return redirect(route('home'))->with('info', 'Your email is already verified. Welcome back!');
        }

        if ($user->markEmailAsVerified()) {
            event(new Verified($user));
            
            // Set user profile status to active after email verification
            if ($user->profile) {
                $user->profile->update(['status' => UserProfileStatus::Active]);
            }
            
            // Automatically log in the user after successful verification
            Auth::login($user, true);
        }

        return redirect(route('home'))->with('success', 'Your email has been verified successfully! Welcome to your account.');
    }
}
