<?php
namespace App\Http\Controllers\Frontend\Blog;

use App\Http\Controllers\Controller;
use App\Services\Blog\PostCategoryService;
use App\Services\Blog\PostService;
use App\Services\Blog\PostTagService;
use App\Services\User\UserService;
use Illuminate\Http\Request;
use RalphJSmit\Laravel\SEO\Support\SEOData;

class PostController extends Controller
{
    protected $postService;
    protected $postCategoryService;
    protected $postTagService;
    protected $userService;

    public function __construct(PostService $postService, PostCategoryService $postCategoryService, PostTagService $postTagService, UserService $userService)
    {
        $this->postService         = $postService;
        $this->postCategoryService = $postCategoryService;
        $this->postTagService      = $postTagService;
        $this->userService         = $userService;
    }

    public function index(Request $request)
    {
        $posts       = $this->postService->getAll()->published()->paginate(10);
        $page        = $request->attributes->get('page');
        $seoData     = $request->attributes->get('seoData');
        $data        = [];
        $breadcrumbs = [
            'name'   => 'journal',
            'params' => [],
        ];
        // dump($breadcrumbs);
        return view('blog.blog-list', compact('posts', 'breadcrumbs', 'data', 'seoData', 'page'));
    }

    public function search(Request $request)
    {
        // Accept both 'search' and 'q' keys (blog-list passes 'search' while legacy may use 'q')
        $q        = $request->query('search', $request->query('q'));
        $category = $request->input('category');
        $page     = $request->attributes->get('page');

        $seoData = $request->attributes->get('seoData');
        if (! $q) {
            $title = "<span class='font-normal'>Search results</span>";
        } else {
            $title = "<span class='font-normal'>Search Results for:</span> \"$q\"";
        }
        $breadcrumbs = ['name' => 'journals-search', 'params' => ['query' => $q, 'category' => $category], 'page' => $page];

        // Retrieve page from middleware for SEO and heading content
        $page    = $request->attributes->get('page');
        $seoData = $request->attributes->get('seoData');
        return view('blog.blog-search', compact('breadcrumbs', 'page'));
    }

    public function category(PostCategoryService $postCategoryService, $slug)
    {
        $category = $postCategoryService->findBySlug($slug);
        $posts    = $this->postService->getAll(['category' => $category->slug])->published()->count();
        if (! $category) {
            abort(404);
        }
        $title        = "<span class='font-normal'>" . $category->name . "</span>  Expert Articles & Insights";
        $subtitle     = "Explore our collection of expert articles on " . $category->name . ", covering design, maintenance, and innovation.";
        $search_title = "Latest Articles in " . $category->name . ":";
        $pageData     = [
            'page_title'    => $title,
            'page_subtitle' => $subtitle,
            'search_title'  => $search_title,
            'sorting'       => true,
            'categorySlug'  => $category->slug,
            'category'      => $category,
            'author'        => '',
            'tag'           => '',
            'query'         => '',
            // 'post_count'    => $posts,
        ];
        $seoData     = $category->getDynamicSEOData();
        $breadcrumbs = ['name' => 'journal-category', 'params' => ['category' => $category]];
        return view('blog.blog-category', compact('category', 'pageData', 'seoData', 'breadcrumbs'));
    }

    public function tag(PostTagService $postTagService, $slug)
    {
        $tag = $postTagService->findBySlug($slug);
        if (! $tag) {
            abort(404);
        }
        $posts = $this->postService->getAll(['tag' => $tag->slug])->published()->count();
        dump($posts);
        $title        = "<span class='font-normal'>" . $tag->name . "</span>  Expert Articles & Insights";
        $subtitle     = "Explore our collection of expert articles on " . $tag->name . ", covering design, maintenance, and innovation.";
        $search_title = "Latest Articles in " . $tag->name . ":";
        $pageData     = [
            'page_title'    => $title,
            'page_subtitle' => $subtitle,
            'search_title'  => $search_title,
            'sorting'       => true,
            'found_posts'   => $posts,
            'tag'           => $tag->slug,
            'category'      => '',
            'author'        => '',
            'query'         => '',
        ];
        $seoData = $tag->getDynamicSEOData();

        $breadcrumbs = ['name' => 'blog-tag', 'params' => ['tag' => $tag]];
        return view('blog.blog-tag', compact('seoData', 'tag', 'breadcrumbs', 'pageData'));
    }

    public function author(UserService $userService, $slug)
    {
        $author = $userService->findBySlug($slug);
        if (! $author) {
            abort(404);
        }
        $posts   = $this->postService->getAll(['author' => $author->id])->published()->count();
        $seoData = new SEOData(
            title: $author->name . " - ESP Marketplace Journal",
            description: "Explore our collection of expert articles by " . $author->name,
        );
        $breadcrumbs  = ['name' => 'blog-author', 'params' => ['author' => $author]];
        $title        = "<span class='font-normal'>" . $author->name . "</span>  Expert Articles & Insights";
        $subtitle     = "Explore our collection of expert articles by " . $author->name;
        $search_title = "Latest Articles by " . $author->name . ":";
        $pageData     = [
            'page_title'    => $title,
            'page_subtitle' => $subtitle,
            'search_title'  => $search_title,
            'sorting'       => true,
            'found_posts'   => $posts,
            'author'        => $author->slug,
            'category'      => '',
            'tag'           => '',
            'query'         => '',
        ];
        return view('blog.blog-author', compact('seoData', 'author', 'breadcrumbs', 'pageData'));
    }

    public function show($slug)
    {
        $post   = $this->postService->findBySlug($slug);
        $author = $post->author;
        if (! $post) {
            abort(404);
        }
        $reviewsStats = $this->userService->getUserReviewsStats($author->id);

        $breadcrumbs = ['name' => 'blog-detail', 'params' => ['post' => $post]];
        return view('blog.show', compact('post', 'breadcrumbs', 'reviewsStats'));
    }
}
