<?php

namespace App\Http\Controllers\Frontend\Member;

use App\Http\Controllers\Controller;
use App\Services\User\UserService;
use App\Services\User\UserVideoService;
use RalphJSmit\Laravel\SEO\Support\SEOData;
use RalphJSmit\Laravel\SEO\Support\Schema\Breadcrumb;
use App\Enums\MembershipType;
use Illuminate\Support\Facades\Request;
// use Illuminate\Http\Request;

class MemberController extends Controller
{
    protected $userService;
    protected $userVideoService;

    public function __construct(UserService $userService, UserVideoService $userVideoService)
    {
        $this->userService = $userService;
        $this->userVideoService = $userVideoService;
    }

    public function index(Request $request)
    {
        // Get page data shared from pageSeoMiddleware
        $page = Request::instance()->attributes->get('page');
        $seoData = Request::instance()->attributes->get('seoData');
        $data = [];
        $viewType = 'list';
        if ($page) {
            $data = $page->data;
            $viewType = !empty($data['member_list_template']['default_view']) ? $data['member_list_template']['default_view'] : 'list';
        }
        if (Request::filled('view_type')) {
            $viewType = Request::get('view_type');
        }
        $defaultView = ($viewType == 'list') ? 'member.list-view' : 'member.map-view';
        $breadcrumbs = [
            'name' => 'member-list',
            'params' => [],
        ];

        return view($defaultView, compact('breadcrumbs', 'data', 'seoData', 'viewType', 'page'));
    }
    public function show($slug)
    {
        $user = $this->userService->findBySlug($slug);

        if (!$user) {
            abort(404);
        }

        $seoData = new SEOData(
            title: $user->name . ' - Earth Services Pro',
            description: $user->bio,
        );
        $breadcrumbs = [
            'name' => 'member',
            'params' => ['member' => $user],
        ];
        $reviewsStats = $this->userService->getUserReviewsStats($user->id);
        $tabs = $this->getMemberTabs($user);
        $activeTab = Request::has('tab') ? Request::get('tab') : 'overview';
        return view('member.profile', compact('user', 'seoData', 'breadcrumbs', 'reviewsStats', 'tabs', 'activeTab'));
    }

    public function showVideo($uuid)
    {
        $video = $this->userVideoService->findByUuid($uuid);
        if (!$video) {
            abort(404);
        }
        $video->increment('views');
        return view('member.video-detail', compact('video'));
    }
    public function getMemberTabs($user)
    {
        $membershipType = $user->membershipType->value;
        $freeTabs = [
            'profile-services',
            'profile-classifieds',
            'profile-reviews',
            'profile-projects',
            'profile-equipment',
            'profile-articles',
            'profile-team',
            'profile-video',
        ];

        $tabs = collect([
            'services' => [
                'order' => 2,
                'label' => 'Services',
                'id' => 'profile-services',
                'lazy' => true,
                'component' => 'frontend.member.profile-tabs.service-tab',
                'params' => ['userId'],
            ],
            'reviews' => [
                'order' => 3,
                'label' => 'Reviews',
                'id' => 'profile-reviews',
                'lazy' => true,
                'component' => 'frontend.member.profile-tabs.review-tab',
                'params' => ['userId', 'reviewsStats'],
            ],
            'projects' => [
                'order' => 4,
                'label' => 'Projects',
                'id' => 'profile-projects',
                'lazy' => true,
                'component' => 'frontend.member.profile-tabs.project-tab',
                'params' => ['userId'],
            ],
            'equipments' => [
                'order' => 5,
                'label' => 'Equipment',
                'id' => 'profile-equipment',
                'lazy' => true,
                'component' => 'frontend.member.profile-tabs.equipment-tab',
                'params' => ['userId'],
            ],
            'articles' => [
                'order' => 6,
                'label' => 'Articles',
                'id' => 'profile-articles',
                'lazy' => true,
                'component' => 'frontend.member.profile-tabs.article-tab',
                'params' => ['userId'],
            ],
            'classifieds' => [
                'order' => 7,
                'label' => 'Classifieds',
                'id' => 'profile-classifieds',
                'lazy' => true,
                'component' => 'frontend.member.profile-tabs.classified-tab',
                'params' => ['userId'],
            ],
            'jobs' => [
                'order' => 8,
                'label' => 'Jobs',
                'id' => 'profile-jobs',
                'lazy' => true,
                'component' => 'frontend.member.profile-tabs.job-tab',
                'params' => ['userId'],
            ],
            'team' => [
                'order' => 9,
                'label' => 'Team',
                'id' => 'profile-team',
                'lazy' => true,
                'component' => 'frontend.member.profile-tabs.team-tab',
                'params' => ['userId'],
            ],
            'videos' => [
                'order' => 10,
                'label' => 'Videos',
                'id' => 'profile-video',
                'lazy' => true,
                'component' => 'frontend.member.profile-tabs.video-tab',
                'params' => ['userId'],
            ],
        ]);
        if ($membershipType == MembershipType::Free->value) {
            $tabs = $tabs->filter(function ($tab) use ($freeTabs) {
                return in_array($tab['id'], $freeTabs);
            });
        }
        return $tabs;
    }
}
