<?php

namespace App\Http\Controllers;

use App\Services\Credit\CreditPurchaseService;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Log;
use Laravel\Cashier\Http\Controllers\WebhookController as CashierWebhookController;

class WebhookController extends CashierWebhookController
{
    protected CreditPurchaseService $creditPurchaseService;

    public function __construct(CreditPurchaseService $creditPurchaseService)
    {
        $this->creditPurchaseService = $creditPurchaseService;
    }

    /**
     * Handle checkout session completed webhook
     */
    public function handleCheckoutSessionCompleted(array $payload): Response
    {
        try {
            $checkoutSession = $payload['data']['object'];
            $sessionId = $checkoutSession['id'];

            // Process the checkout session completion
            $result = $this->creditPurchaseService->handleCheckoutSessionCompleted($sessionId);

            return new Response('Webhook handled', 200);
        } catch (\Exception $e) {
            return new Response('Webhook error', 500);
        }
    }

    /**
     * Handle payment intent succeeded webhook
     */
    public function handlePaymentIntentSucceeded(array $payload): Response
    {
        try {
            $paymentIntent = $payload['data']['object'];
            $metadata = $paymentIntent['metadata'] ?? [];

            // If this payment is linked to a purchase in our system
            if (isset($metadata['payment_id'])) {
                $paymentId = $metadata['payment_id'];

                // Update payment status to completed
                $payment = \App\Models\Membership\Payment::find($paymentId);
                if ($payment) {
                    $payment->update([
                        'status' => \App\Enums\PaymentStatus::Paid->value,
                        'service_response' => json_encode([
                            'stripe_payment_intent' => $paymentIntent,
                            'status' => 'succeeded',
                            'timestamp' => now()->toISOString(),
                        ]),
                    ]);

                    Log::info('Payment succeeded', ['payment_id' => $paymentId]);
                }
            }

            return new Response('Webhook handled', 200);
        } catch (\Exception $e) {
            // Log the error but return 200 to acknowledge receipt
            Log::error('Error handling payment success webhook: ' . $e->getMessage(), [
                'exception' => $e,
                'payload' => $payload
            ]);
            return new Response('Webhook received', 200);
        }
    }

    /**
     * Handle payment failed webhook
     */
    public function handlePaymentIntentPaymentFailed(array $payload): Response
    {
        try {
            Log::info('Payment failed', ['payload' => $payload]);
            $paymentIntent = $payload['data']['object'];
            $metadata = $paymentIntent['metadata'] ?? [];

            // If this payment is linked to a purchase in our system
            if (isset($metadata['payment_id'])) {
                $paymentId = $metadata['payment_id'];

                // Update payment status to failed
                $payment = \App\Models\Membership\Payment::find($paymentId);
                if ($payment) {
                    $payment->update([
                        'status' => \App\Enums\PaymentStatus::Failed->value,
                        'service_response' => json_encode([
                            'stripe_payment_intent' => $paymentIntent,
                            'error' => $paymentIntent['last_payment_error'] ?? 'Payment failed',
                            'timestamp' => now()->toISOString(),
                        ]),
                    ]);

                    // Notify user about the failed payment if needed
                    $user = $payment->user;
                    if ($user) {
                        $user->notify(new \App\Notifications\CreditPurchaseFailed(
                            $payment->purchase,
                            $payment
                        ));
                    }

                    Log::info('Payment failed', ['payment_id' => $paymentId]);
                }
            }

            return new Response('Payment failure handled', 200);
        } catch (\Exception $e) {
            // Log the error but return 200 to acknowledge receipt
            Log::error('Error handling payment failure webhook: ' . $e->getMessage(), [
                'exception' => $e,
                'payload' => $payload
            ]);
            return new Response('Webhook received', 200);
        }
    }
}
