<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use App\Contracts\Previewable;

class PreviewMiddleware
{
    protected $modelMap = [
        'blog' => \App\Models\Blog\Post::class,
        'classified' => \App\Models\Classified\Classified::class,
        'job' => \App\Models\Job\Job::class,
        'page' => \App\Models\Page::class,
    ];

    public function handle(Request $request, Closure $next)
    {
        $path = $request->segment(1);
        $slug = $request->route('slug');
        $isPreviewMode = $request->has('preview');
        
        // Get the corresponding model class
        $modelClass = $this->modelMap[$path] ?? null;
        
        if (!$modelClass || !class_exists($modelClass)) {
            return $next($request);
        }

        /** @var Model $model */
        $model = $modelClass::where('slug', $slug)->first();
        
        if (!$model) {
            abort(404);
        }

        // Check if model implements Previewable interface
        if ($this->isPreviewable($model)) {
            // For preview mode, check authentication and permissions
            if ($isPreviewMode) {
                if (!$request->user()) {
                    abort(401); // Unauthorized if not logged in
                }
                
                if (!$this->canPreview($request, $model)) {
                    abort(403); // Forbidden if user doesn't have permission
                }
                
                // Allow preview regardless of publication status
                return $next($request);
            }

            // For non-preview mode, only allow access to published content
            if (!$model->isPublished()) {
                abort(404);
            }
        }

        return $next($request);
    }

    protected function isPreviewable($model): bool
    {
        return $model instanceof Previewable;
    }

    protected function canPreview(Request $request, $model): bool
    {
        $user = $request->user();
        if (!$user) {
            return false;
        }
    
        // Check for specific permission based on model type
        $permission = 'view_' . Str::plural(Str::snake(class_basename($model)));
        return $user->can($permission);
    }
} 