<?php

namespace App\Http\Resources\V1\Lead\LeadForm;

use App\Enums\LeadFormLayout;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class LeadFormDetailResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        $baseData = [
            'id' => $this->uuid,
            'title' => $this->title,
            'description' => $this->description,
            'match_criteria' => $this->match_criteria?->value,
            'reception_mode' => $this->reception_mode?->value,
            'enable_captcha' => $this->enable_captcha,
            'is_active' => $this->is_active,
            'submit_message' => $this->submit_message,
            'form_layout' => $this->form_layout?->value,
            'created_at' => $this->created_at?->toIso8601String(),
            'updated_at' => $this->updated_at?->toIso8601String(),
        ];

        // Conditional fields/steps based on form layout
        if ($this->relationLoaded('formFields')) {
            if ($this->form_layout === LeadFormLayout::Wizard && ! empty($this->form_steps)) {
                // Wizard layout: Group fields by steps
                $baseData['steps'] = $this->formatWizardSteps();
            } else {
                // Simple layout: Flat fields array
                $baseData['fields'] = LeadFormFieldResource::collection($this->formFields);
            }
        }

        // Statistics
        $baseData['visits_count'] = $this->when(isset($this->visits_count), $this->visits_count);
        $baseData['leads_count'] = $this->when(isset($this->leads_count), $this->leads_count);

        // Author
        $baseData['author'] = $this->when($this->relationLoaded('author'), function () {
            return [
                'id' => $this->author->id,
                'name' => $this->author->name,
                'email' => $this->author->email,
            ];
        });

        // Member categories
        $baseData['member_categories'] = $this->when($this->relationLoaded('memberCategories'), function () {
            return $this->memberCategories->map(function ($category) {
                return [
                    'id' => $category->id,
                    'name' => $category->name,
                    'credits' => $category->pivot->credits ?? null,
                    'radius' => $category->pivot->radius ?? null,
                ];
            });
        });

        return $baseData;
    }

    /**
     * Format wizard steps with their associated fields
     */
    protected function formatWizardSteps(): array
    {
        // Group fields by form_step UUID
        $fieldsByStep = $this->formFields->groupBy(function ($field) {
            return $field->pivot->form_step ?? 'unassigned';
        });

        // Map each step from form_steps
        return collect($this->form_steps)->map(function ($step) use ($fieldsByStep) {
            $stepFields = $fieldsByStep->get($step['uuid'] ?? null, collect());

            return [
                'uuid' => $step['uuid'] ?? null,
                'name' => $step['name'] ?? null,
                'title' => $step['title'] ?? null,
                'subtitle' => $step['subtitle'] ?? null,
                'image' => $step['image'] ? url('storage/'.$step['image']) : null,
                'is_active' => $step['is_active'] ?? true,
                'fields' => LeadFormFieldResource::collection($stepFields),
            ];
        })->toArray();
    }
}
