<?php

namespace App\Http\Resources\V1\Lead\LeadForm;

use App\Enums\FieldDataSource;
use App\Enums\FileUploadType;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class LeadFormFieldResource extends JsonResource
{
    /**
     * Get enum options from an enum class
     * Returns key-value array format like ["value" => "Label"] for consistency with manual options
     */
    private function getEnumOptions(string $enumClass): array
    {
        if (! enum_exists($enumClass)) {
            return [];
        }

        $options = [];
        $cases = $enumClass::cases();

        foreach ($cases as $case) {
            // Check if the enum has a getLabel() method
            $label = method_exists($case, 'getLabel') ? $case->getLabel() : $case->name;

            // Get the value (for backed enums) or use the name (for unit enums)
            $value = $case instanceof \BackedEnum ? $case->value : $case->name;

            // Return as key-value pair
            $options[$value] = $label;
        }

        return $options;
    }

    /**
     * Get dependent fields grouped by parent option value
     */
    private function getDependentFields(): array
    {
        if (! isset($this->pivot) || ! $this->pivot->id) {
            return [];
        }

        // Get all child fields that depend on this field
        $childFields = \App\Models\Lead\FieldLeadForm::query()
            ->where('lead_form_id', $this->pivot->lead_form_id)
            ->whereNotNull('conditional_logic')
            ->get()
            ->filter(function ($field) {
                return ! empty($field->conditional_logic['enabled']) &&
                       ! empty($field->conditional_logic['parent_field_id']) &&
                       $field->conditional_logic['parent_field_id'] == $this->pivot->id;
            });

        if ($childFields->isEmpty()) {
            return [];
        }

        // Group child fields by parent option value
        $dependentFieldsGrouped = [];

        foreach ($childFields as $childField) {
            $parentValue = $childField->conditional_logic['parent_option_value'] ?? null;

            if ($parentValue === null) {
                continue;
            }

            // Load the field relationship if not already loaded
            if (! $childField->relationLoaded('field')) {
                $childField->load('field');
            }

            // Build the child field data structure
            $childFieldData = [
                'id' => $childField->id, // Pivot table ID
                'uuid' => $childField->field->uuid ?? null, // Field's UUID from related model
                'field_name' => $childField->field->field_name ?? null,
                'field_id' => $childField->field->field_id ?? null, // Field identifier from related model
                'field_type' => $childField->field->field_type?->value ?? null,
                'data_type' => $childField->field->data_type?->value ?? null,
                'is_default' => $childField->field->is_default ?? false,
                'config' => $this->buildChildFieldConfig($childField),
            ];

            // Recursively get dependent fields for this child
            $nestedDependents = $this->getNestedDependentFields($childField);
            if (! empty($nestedDependents)) {
                $childFieldData['dependent_fields'] = $nestedDependents;
            }

            // Add to the grouped array
            if (! isset($dependentFieldsGrouped[$parentValue])) {
                $dependentFieldsGrouped[$parentValue] = [];
            }

            $dependentFieldsGrouped[$parentValue][] = $childFieldData;
        }

        return $dependentFieldsGrouped;
    }

    /**
     * Get nested dependent fields for a child field
     */
    private function getNestedDependentFields(\App\Models\Lead\FieldLeadForm $parentField): array
    {
        $childFields = \App\Models\Lead\FieldLeadForm::query()
            ->where('lead_form_id', $parentField->lead_form_id)
            ->whereNotNull('conditional_logic')
            ->get()
            ->filter(function ($field) use ($parentField) {
                return ! empty($field->conditional_logic['enabled']) &&
                       ! empty($field->conditional_logic['parent_field_id']) &&
                       $field->conditional_logic['parent_field_id'] == $parentField->id;
            });

        if ($childFields->isEmpty()) {
            return [];
        }

        $dependentFieldsGrouped = [];

        foreach ($childFields as $childField) {
            $parentValue = $childField->conditional_logic['parent_option_value'] ?? null;

            if ($parentValue === null) {
                continue;
            }

            if (! $childField->relationLoaded('field')) {
                $childField->load('field');
            }

            $childFieldData = [
                'id' => $childField->id, // Pivot table ID
                'uuid' => $childField->field->uuid ?? null, // Field's UUID from related model
                'field_name' => $childField->field->field_name ?? null,
                'field_id' => $childField->field->field_id ?? null, // Field identifier from related model
                'field_type' => $childField->field->field_type?->value ?? null,
                'data_type' => $childField->field->data_type?->value ?? null,
                'is_default' => $childField->field->is_default ?? false,
                'config' => $this->buildChildFieldConfig($childField),
            ];

            if (! isset($dependentFieldsGrouped[$parentValue])) {
                $dependentFieldsGrouped[$parentValue] = [];
            }

            $dependentFieldsGrouped[$parentValue][] = $childFieldData;
        }

        return $dependentFieldsGrouped;
    }

    /**
     * Process file validation settings for file upload fields
     */
    private function processFileValidation(?array $fieldData, string $fieldType): ?array
    {
        // Only process if field type is file and validation settings exist
        if ($fieldType !== 'file' || empty($fieldData['file_validation'])) {
            return null;
        }

        $validation = $fieldData['file_validation'];
        $allowedTypes = $validation['allowed_types'] ?? ['image'];

        // Convert string types to FileUploadType enums
        $fileTypes = collect($allowedTypes)
            ->map(fn ($type) => FileUploadType::tryFrom($type))
            ->filter()
            ->all();

        if (empty($fileTypes)) {
            // Default to image if no valid types
            $fileTypes = [FileUploadType::Image];
        }

        return [
            'allowed_types' => array_map(fn ($type) => $type->value, $fileTypes),
            'max_size_kb' => $validation['max_size_kb'] ?? 2048,
            'max_files' => $validation['max_files'] ?? 5,
            'accept' => FileUploadType::getCombinedAcceptAttribute($fileTypes),
            'mime_types' => FileUploadType::getCombinedMimeTypes($fileTypes),
            'extensions' => FileUploadType::getCombinedExtensions($fileTypes),
        ];
    }

    /**
     * Build config array for a child field
     */
    private function buildChildFieldConfig(\App\Models\Lead\FieldLeadForm $childField): array
    {
        $dataSource = $childField->data_source ?? null;
        $fieldOptions = $childField->field_options ?? null;

        if ($dataSource === FieldDataSource::Enum->value && is_string($fieldOptions) && class_exists($fieldOptions)) {
            $fieldOptions = $this->getEnumOptions($fieldOptions);
        } elseif (is_string($fieldOptions) && class_exists($fieldOptions) && enum_exists($fieldOptions)) {
            $fieldOptions = $this->getEnumOptions($fieldOptions);
        } elseif (is_string($fieldOptions)) {
            $decoded = json_decode($fieldOptions, true);
            $fieldOptions = $decoded && is_array($decoded) ? $decoded : null;
        }

        $fieldData = $childField->field_data ?? null;
        if (is_string($fieldData)) {
            $fieldData = json_decode($fieldData, true);
        }

        // Process file validation settings
        $fileValidation = $this->processFileValidation($fieldData, $childField->field->field_type?->value ?? '');

        $config = [
            'field_label' => $childField->field_label ?? null,
            'field_placeholder' => $childField->field_placeholder ?? null,
            'toggle_yes_text' => $childField->toggle_yes_text ?? null,
            'toggle_no_text' => $childField->toggle_no_text ?? null,
            'field_columns' => $childField->field_columns ?? 12,
            'help_text' => $childField->help_text ?? null,
            'field_options' => $fieldOptions,
            'field_data' => $fieldData,
            'data_columns' => $fieldData['data_columns'] ?? null,
            'container_class' => $fieldData['container_class'] ?? null,
            'container_id' => $fieldData['container_id'] ?? null,
            'is_required' => $childField->is_required ?? false,
            'is_visible' => $childField->is_visible ?? true,
            'field_order' => $childField->field_order ?? 0,
        ];

        // Add file validation if available
        if ($fileValidation) {
            $config['file_validation'] = $fileValidation;
        }

        return $config;
    }

    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        $dependentFields = $this->getDependentFields();

        return [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'field_name' => $this->field_name,
            'field_id' => $this->field_id,
            'field_type' => $this->field_type?->value,
            'data_type' => $this->data_type?->value,
            'is_default' => $this->is_default,

            // Pivot data (configuration in form)
            'config' => $this->when(isset($this->pivot), function () {
                $dataSource = $this->pivot->data_source ?? null;
                $fieldOptions = $this->pivot->field_options ?? null;

                // Handle enum data source - check if it's enum type OR if field_options is an enum class path
                if ($dataSource === FieldDataSource::Enum->value && is_string($fieldOptions) && class_exists($fieldOptions)) {
                    // Load the enum class and extract cases as key-value pairs
                    $fieldOptions = $this->getEnumOptions($fieldOptions);
                } elseif (is_string($fieldOptions) && class_exists($fieldOptions) && enum_exists($fieldOptions)) {
                    // Fallback: if field_options is an enum class path but data_source wasn't set (old data)
                    $fieldOptions = $this->getEnumOptions($fieldOptions);
                } elseif (is_string($fieldOptions)) {
                    // Parse field_options if it's a JSON string - keep as key-value pairs
                    $decoded = json_decode($fieldOptions, true);
                    $fieldOptions = $decoded && is_array($decoded) ? $decoded : null;
                } elseif (is_array($fieldOptions)) {
                    // Already an array, keep as-is (key-value pairs)
                    $fieldOptions = $fieldOptions;
                }

                // Parse field_data if it's a JSON string
                $fieldData = $this->pivot->field_data ?? null;
                if (is_string($fieldData)) {
                    $fieldData = json_decode($fieldData, true);
                }

                // Process file validation settings
                $fileValidation = $this->processFileValidation($fieldData, $this->field_type?->value ?? '');

                $config = [
                    'field_label' => $this->pivot->field_label ?? null,
                    'field_placeholder' => $this->pivot->field_placeholder ?? null,
                    'toggle_yes_text' => $this->pivot->toggle_yes_text ?? null,
                    'toggle_no_text' => $this->pivot->toggle_no_text ?? null,
                    'field_columns' => $this->pivot->field_columns ?? 12,
                    'help_text' => $this->pivot->help_text ?? null,
                    'field_options' => $fieldOptions,
                    'field_data' => $fieldData,
                    'data_columns' => $fieldData['data_columns'] ?? null,
                    'container_class' => $fieldData['container_class'] ?? null,
                    'container_id' => $fieldData['container_id'] ?? null,
                    'is_required' => $this->pivot->is_required ?? false,
                    'is_visible' => $this->pivot->is_visible ?? true,
                    'field_order' => $this->pivot->field_order ?? 0,
                ];

                // Add file validation if available
                if ($fileValidation) {
                    $config['file_validation'] = $fileValidation;
                }

                return $config;
            }),

            // Dependent fields grouped by parent option value
            'dependent_fields' => ! empty($dependentFields) ? $dependentFields : null,

            'created_at' => $this->created_at?->toIso8601String(),
            'updated_at' => $this->updated_at?->toIso8601String(),
        ];
    }
}
