<?php

namespace App\Livewire\Dashboard\AccountSubscription;

use App\Services\User\UserSubscriptionService;
use Livewire\Component;
use Filament\Notifications\Notification;

class AccountSubscription extends Component
{

    protected $userSubscriptionService;

    protected $user;
    public $subscription;
    public $subscriberPlan;
    public $creditBalance;

    public function boot(UserSubscriptionService $userSubscriptionService)
    {
        $this->userSubscriptionService = $userSubscriptionService;
    }

    public function mount($user)
    {
        $this->user = $user;
        $this->getSubscriptionDetails();

        // Check for Stripe redirect parameters
        $this->handleStripeRedirect();
    }

    private function handleStripeRedirect()
    {
        // Handle successful payment
        if (request()->has('success')) {
            // Refresh user data to get updated credit balance
            $this->user->refresh();
            $this->getSubscriptionDetails();

            Notification::make()
                ->title('Payment Successful!')
                ->body('Your credit purchase has been completed successfully. Your credits have been added to your account.')
                ->success()
                ->persistent()
                ->send();
        }
    }

    public function getSubscriptionDetails()
    {
        $userId = $this->user->id;
        $this->subscription = $this->userSubscriptionService->getByUserId($userId);
        $this->subscriberPlan = $this->subscription?->membershipPlan;
        $this->creditBalance = $this->user->profile?->credit_balance ?? 0;
    }

    /**
     * Check if user has an active subscription
     */
    public function hasActiveSubscription(): bool
    {
        return $this->subscription !== null && $this->subscription->isActive();
    }

    /**
     * Get the subscription status message
     */
    public function getSubscriptionStatusMessage(): string
    {
        if (!$this->subscription) {
            return __('No active subscription');
        }

        if ($this->subscription->isExpired()) {
            return __('Subscription expired');
        }

        if ($this->subscription->isActive()) {
            return __('Active subscription');
        }

        return __('Subscription status unknown');
    }

    public function render()
    {
        return view('livewire.dashboard.account-subscription.account-subscription');
    }
}
