<?php

namespace App\Livewire\Dashboard\AccountSubscription;

use Livewire\Component;

use App\Models\Shop\Product;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use App\Services\User\UserSubscriptionService;
use Illuminate\Support\Facades\Auth;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use App\Services\Purchase\PurchaseService;
use App\Enums\PurchaseItemType;
use App\Services\Credit\CreditService;
use App\Enums\MembershipPaymentTerm;
use App\Enums\SubscriptionStatus;
use Filament\Tables\Actions\BulkAction;
use Filament\Tables\Actions\BulkActionGroup;
use Livewire\Attributes\Url;
use App\Services\User\MemberPlanService;

class TransectionHistory extends Component implements HasForms, HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;

    protected $userSubscriptionService;
    protected $creditService;
    protected $purchaseService;
    protected $memberPlanService;

    public $activeTab;
    public $subscriptionTab;
    public $transectionsTab;

    public $memberPlans = [];

    #[Url]
    public $search;

    #[Url]
    public $memberPlan;

    public $dateRange;

    public $startDate;
    public $endDate;

    public function boot(
        UserSubscriptionService $userSubscriptionService,
        CreditService $creditService,
        PurchaseService $purchaseService,
        MemberPlanService $memberPlanService,
    ) {
        $this->userSubscriptionService = $userSubscriptionService;
        $this->creditService = $creditService;
        $this->purchaseService = $purchaseService;
        $this->memberPlanService = $memberPlanService;
    }

    public function mount()
    {
        $this->activeTab = PurchaseItemType::Subscription->value;

        // get member plans
        if ($this->activeTab === PurchaseItemType::Subscription->value) {
            $this->getMemberPlans();
        }
    }

    public function updated($property, $value)
    {
        if (in_array($property, ['memberPlan', 'dateRange', 'startDate', 'endDate'])) {
            $this->applyFilter();
        }
    }

    public function switchTab($tab): void
    {
        $this->activeTab = $tab;
        if ($tab == 'credit') {
            $this->reset(['memberPlan']);
        }
        // Reset table state when switching tabs
        $this->resetTable();
    }

    public function table(Table $table): Table
    {
        switch ($this->activeTab) {
            case PurchaseItemType::Subscription->value:
                $table = self::subscriptionTable($table);
                break;
            case PurchaseItemType::Credit->value:
                $table = self::transectionsTable($table);
                break;
            default:
                $table = self::subscriptionTable($table);
                break;
        }
        return $table;
    }

    protected function subscriptionTable(Table $table): Table
    {
        return $table
            ->query($this->getSubscriptionQuery())
            ->columns([
                TextColumn::make('package_title')
                    ->label('Package')
                    ->getStateUsing(function ($record) {
                        if ($record->item_type === PurchaseItemType::Subscription) {
                            $subscription = $record->purchaseable;
                            return $subscription?->membershipPlan?->package_title ?? 'Unknown Plan';
                        }
                        return $record->purchaseable_type ;
                    })
                    ->sortable(),
                TextColumn::make('amount')
                    ->label('Amount')
                    ->getStateUsing(function ($record) {
                        if ($record->item_type) {
                            $subscription = $record->purchaseable;
                            $paymentTerm = $subscription->payment_term ?? 0;
                            $price = 0;
                            switch ($paymentTerm) {
                                case MembershipPaymentTerm::Monthly:
                                    $price = $subscription->membershipPlan->price_monthly ?? 0;
                                    break;
                                case MembershipPaymentTerm::Quarterly:
                                    $price = $subscription->membershipPlan->price_quarterly ?? 0;
                                    break;
                                case MembershipPaymentTerm::SemiYearly:
                                    $price = $subscription->membershipPlan->price_semi_yearly ?? 0;
                                    break;
                                case MembershipPaymentTerm::Yearly:
                                    $price = $subscription->membershipPlan->price_yearly ?? 0;
                                    break;
                                default:
                                    $price = 0;
                                    break;
                            }
                        }
                        return $price;
                    })
                    ->formatStateUsing(
                        function ($state) {
                            return esp_currency_formated(price: $state, currency: 'USD', precision: 2);
                        }
                    )
                    ->sortable(),
                TextColumn::make('purchase_date')
                    ->label('Date')
                    ->sortable()
                    ->getStateUsing(function ($record) {
                        if ($record->item_type) {
                            $subscription = $record->purchaseable;
                            return $subscription->purchase_date ?? null;
                        }
                        return null;
                    })
                    ->date('M d, Y'),
                TextColumn::make('is_active')
                    ->label('Status')
                    ->sortable()
                    ->badge()
                    ->color(fn($state) => match ($state) {
                        SubscriptionStatus::Active->getLabel() => 'success',
                        SubscriptionStatus::Canceled->getLabel() => 'danger',
                        default => 'gray',
                    })
                    ->getStateUsing(function ($record) {
                        if ($record->item_type === PurchaseItemType::Subscription) {
                            $subscription = $record->purchaseable;
                            $subscriptionStatus = $subscription->status;
                            switch ($subscriptionStatus) {
                                case SubscriptionStatus::Active->value:
                                    $status = SubscriptionStatus::Active->getLabel();
                                    break;
                                case SubscriptionStatus::Canceled->value:
                                    $status = SubscriptionStatus::Canceled->getLabel();
                                    break;
                                default:
                                    $status = SubscriptionStatus::Canceled->getLabel();
                                    break;
                            }
                        }
                        return $status;
                    }),
                TextColumn::make('payment_method')
                    ->label('Payment Method')
                    ->getStateUsing(function ($record) {
                        $subscription = $record->item_type;
                        if ($subscription === PurchaseItemType::Subscription) {
                            $payment = $record->payments->where('purchase_id', $record->id)->first();
                            return $payment->payment_method ?? '-';
                        }
                        return '-';
                    }),
            ])
            ->defaultSort('purchase_date', 'desc')
            ->actions([
                ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->icon('heroicon-o-eye')
                        ->color('primary'),
                    Action::make('edit')
                        ->label('Edit Subscription')
                        ->icon('heroicon-o-pencil')
                        ->color('warning'),
                    Action::make('cancel')
                        ->label('End Subscription')
                        ->icon('heroicon-o-x-mark')
                        ->color('danger')
                        ->requiresConfirmation(),
                ]),
            ])->bulkActions([
                BulkActionGroup::make([
                    BulkAction::make('cancel')
                        ->label('End Subscription')
                        ->icon('heroicon-o-x-mark')
                        ->color('danger')
                        ->requiresConfirmation(),
                ]),
            ]);
    }


    protected function transectionsTable(Table $table): Table
    {
        return $table
            ->query($this->getCreditsQuery())
            ->columns([
                TextColumn::make('amount')
                    ->label('Amount ')
                    ->searchable()
                    ->getStateUsing(function ($record) {
                        $creditType = $record->item_type;
                        if ($creditType === PurchaseItemType::Credit) {
                            $amount = $record->amount;
                            if (!empty($amount)) {
                                return esp_currency_formated(price: $amount, currency: 'USD', precision: 2);
                            }
                        }
                    }),

                TextColumn::make('quantity')
                    ->label('Quantity')
                    ->getStateUsing(function ($record) {
                        $creditType = $record->item_type;
                        $quantity = $record->quantity ?? 'N/A';
                        if ($creditType === PurchaseItemType::Credit) {
                            if (!empty($quantity)) {
                                $quantity = $quantity;
                            }
                        }
                        return $quantity;
                    }),
                TextColumn::make("payment_method")
                    ->label('Payment Method')
                    ->getStateUsing(function ($record) {
                        $creditType = $record->item_type;
                        $method = 'N/A';
                        if ($creditType === PurchaseItemType::Credit) {
                            $payment = $record->payments->where('purchase_id', $record->id)->first();
                            if (!empty($payment)) {
                                $method = $payment->payment_method ?? $method;
                            }
                        }
                        return $method;
                    }),
                TextColumn::make("payments_status")
                    ->label('Payment Status')
                    ->getStateUsing(function ($record) {
                        $payment = $record->latestPayment();
                        $status = 'N/A';
                        if ($payment) {
                            $status = $payment->status;
                        }
                        return $status;
                    }),
                TextColumn::make('created_at')
                    ->label('Date')
                    ->searchable()
                    ->date()
                    ->formatStateUsing(function ($state) {
                        return $state->diffForHumans();
                    }),
            ])
            ->defaultSort('created_at', 'desc')
            ->actions([
                ActionGroup::make([
                    Action::make('view')
                        ->label('View Details')
                        ->icon('heroicon-o-eye')
                        ->color('primary'),
                ]),
            ])->bulkActions([
                BulkActionGroup::make([
                    BulkAction::make('cancel')
                        ->label('Delete')
                        ->icon('heroicon-o-x-mark')
                        ->color('danger')
                        ->requiresConfirmation(),
                ]),
            ]);
    }

    public function getSubscriptionQuery()
    {
        $args = [
            'user_id' => Auth::id(),
            'item_type' => PurchaseItemType::Subscription,
            'memberPlan' => $this->memberPlan,
            'dateRange' => $this->dateRange,
            'startDate' => $this->startDate,
            'endDate' => $this->endDate,
        ];
        return $this->purchaseService->getAll($args);
    }

    public function getCreditsQuery()
    {
        $args = [
            'user_id' => Auth::id(),
            'item_type' => PurchaseItemType::Credit,
            'dateRange' => $this->dateRange,
            'startDate' => $this->startDate,
            'endDate' => $this->endDate,
        ];
        return $this->purchaseService->getAll($args);
    }


    public function getMemberPlans()
    {
        $this->memberPlans = $this->memberPlanService->getAll()->get();
    }

    public function applyFilter()
    {
        $searchParams = [];

        if ($this->activeTab === PurchaseItemType::Subscription->value) {
            $searchParams['memberPlan'] = $this->memberPlan;
        }

        if (!empty($this->dateRange)) {
            if (is_numeric($this->dateRange)) {
                $searchParams['dateRange'] = (int) $this->dateRange;
            } else if ($this->dateRange === 'custom') {
                $searchParams['startDate'] = $this->startDate;
                $searchParams['endDate'] = $this->endDate;
            }
        }

        $this->purchaseService->getAll($searchParams);

        $this->resetTable();
    }


    public function render()
    {
        return view('livewire.dashboard.account-subscription.transection-history');
    }
}
