<?php

namespace App\Livewire\Dashboard\Applicants;

use App\Models\Shop\Product;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use Illuminate\Contracts\View\View;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Services\Job\JobApplicationService;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Livewire\Attributes\Locked;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use App\Enums\JobApplicationStatus;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\BulkAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TrashedFilter;
use Illuminate\Support\Collection;
use Livewire\Attributes\Url;
use Filament\Tables\Columns\ViewColumn;
use App\Models\Job\JobApplication;


#[Layout('layouts.dashboard.dashboard-app')]
class Applicants extends Component implements HasTable, HasForms
{
    use InteractsWithTable, InteractsWithForms;

    #[Locked]
    protected JobApplicationService $jobApplicationService;

    #[Url]
    public $status = null;

    public function boot(JobApplicationService $jobApplicationService)
    {
        $this->jobApplicationService = $jobApplicationService;
    }

    public function getTableQuery(): Builder
    {
        $args = [
            'user_id' => Auth::id(),
        ];

        if ($this->status) {
            $args['application_status'] = $this->status;
        }

        $query = $this->jobApplicationService->getAll($args);

        return $query;
    }

    public function getTotalCount(): int
    {
        $args = [
            'user_id' => Auth::id(),
        ];

        return $this->jobApplicationService->getAll($args)->count();
    }

    public function getStatusCount(JobApplicationStatus $status): int
    {
        $args = [
            'user_id' => Auth::id(),
            'application_status' => $status->value,
        ];

        return $this->jobApplicationService->getAll($args)->count();
    }

    public function getAllStatusCounts(): array
    {
        $counts = [];
        foreach (JobApplicationStatus::cases() as $status) {
            $counts[$status->value] = $this->getStatusCount($status);
        }
        return $counts;
    }

    public function table(Table $table)
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                TextColumn::make('job.job_title')
                    ->label('Title')
                    ->limit(20)
                    ->tooltip(fn($state): string => $state)
                    ->searchable()
                    ->default('N/A'),
                ViewColumn::make('job-author-info')
                    ->label('Applicant')
                    ->view('tables.columns.job-author-info'),

                TextColumn::make('job.job_industry.industry_name')
                    ->label('Company')
                    ->searchable()
                    ->default('N/A')
                    ->sortable(),
                TextColumn::make('job.job_role.job_role_name')
                    ->label('Role')
                    ->default('N/A')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('application_status')
                    ->label('Status')
                    ->badge()
                    ->sortable(),
                TextColumn::make('created_at')
                    ->label('Published Date')
                    ->default('N/A')
                    ->formatStateUsing(fn($state): string => Carbon::parse($state)->format('d M Y'))
                    ->sortable(),
            ])
            ->filters([
                TrashedFilter::make()
                    ->label('Trashed'),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('view-applicants')
                        ->label('View')
                        ->icon('heroicon-m-eye')
                        ->color('primary')
                        ->modal(true)
                        ->url(function ($record) {
                            return url('dashboard/job/application/' . $record->uuid);
                        })
                        ->slideOver(),

                    Action::make('hired')
                        ->label('Hired')
                        ->requiresConfirmation()
                        ->action(function ($record) {
                            $this->jobApplicationService->updateStatus($record->id, JobApplicationStatus::Hired);
                        })
                        ->icon('heroicon-m-check-circle')
                        ->color('success')
                        ->visible(function ($record) {
                            $applicationStatus = $record->application_status;
                            if ($applicationStatus == JobApplicationStatus::Shortlisted || $applicationStatus == JobApplicationStatus::Pending) {
                                return true;
                            }
                            return false;
                        }),
                    Action::make('shortlisted')
                        ->label('Shortlisted')
                        ->requiresConfirmation()
                        ->action(function ($record) {
                            $this->jobApplicationService->updateStatus($record->id, JobApplicationStatus::Shortlisted);
                        })
                        ->icon('heroicon-m-list-bullet')
                        ->color('warning')
                        ->visible(function ($record) {
                            $applicationStatus = $record->application_status;
                            if ($applicationStatus == JobApplicationStatus::Pending) {
                                return true;
                            }
                            return false;
                        }),
                    Action::make('reject')
                        ->label('Reject')
                        ->requiresConfirmation()
                        ->action(function ($record) {
                            $this->jobApplicationService->updateStatus($record->id, JobApplicationStatus::Rejected);
                        })
                        ->icon('heroicon-m-x-circle')
                        ->color('danger')
                        ->visible(function ($record) {
                            $applicationStatus = $record->application_status;
                            if ($applicationStatus == JobApplicationStatus::Shortlisted || $applicationStatus == JobApplicationStatus::Pending) {
                                return true;
                            }
                            return false;
                        }),
                    DeleteAction::make('delete')
                        ->label('Delete')
                        ->icon('heroicon-m-trash')
                        ->color('danger')
                        ->action(function ($record) {
                            $record->delete();
                        }),
                    RestoreAction::make()
                        ->label('Restore')
                        ->icon('heroicon-m-arrow-path')
                        ->color('warning')
                        ->action(function ($record) {
                            $record->restore();
                        }),
                    ForceDeleteAction::make()
                        ->label('Force Delete')
                        ->icon('heroicon-m-trash')
                        ->color('danger')
                        ->action(function ($record) {
                            $record->forceDelete();
                        }),
                ]),
            ])->bulkActions([
                BulkActionGroup::make([
                    BulkAction::make('delete')
                        ->label('Delete')
                        ->requiresConfirmation()
                        ->icon('heroicon-m-trash')
                        ->color('danger')
                        ->action(fn(Collection $records) => $records->each->delete()),
                    BulkAction::make('restore')
                        ->label('Restore')
                        ->requiresConfirmation()
                        ->icon('heroicon-m-arrow-path')
                        ->color('warning')
                        ->action(fn(Collection $records) => $records->each->restore()),
                    BulkAction::make('forceDelete')
                        ->requiresConfirmation()
                        ->label('Force Delete')
                        ->icon('heroicon-m-trash')
                        ->color('danger')
                        ->action(fn(Collection $records) => $records->each->forceDelete()),
                ]),
            ]);
    }

    public function filterByStatus($status)
    {
        // Convert string to enum if needed
        if (is_string($status)) {
            $this->status = $status;
        } elseif ($status instanceof JobApplicationStatus) {
            $this->status = $status->value;
        } else {
            $this->status = $status;
        }

        $this->resetTable();
    }

    public function clearStatusFilter()
    {
        $this->status = null;
        $this->resetTable();
    }

    public function render()
    {
        return view('livewire.dashboard.applicants.applicants');
    }
}
