<?php
namespace App\Livewire\Dashboard\Article;

use App\Enums\PostStatus;
use App\Models\Blog\Post;
use App\Services\Blog\PostService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Article\ArticleForm;
use Filament\Notifications\Notification;

#[Layout('layouts.dashboard.dashboard-app')]
class CreateArticle extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public ?string $action = null;

    protected PostService $postService;

    public function boot(PostService $postService)
    {
        $this->postService = $postService;
    }

    public function mount(): void
    {
        $this->form->fill([
            'post_status' => PostStatus::Pending->value,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new ArticleForm())->getForm('create')
            )
            ->model(Post::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $this->action = 'draft';
        $this->create();   
    }

    public function create(): void
    {
        $data = $this->form->getState();
        $data['post_status'] = match($this->action) {
            'draft' => PostStatus::Draft->value,
            default => PostStatus::Pending->value
        };

        // Create post using PostService
        $post = $this->postService->create($data);
        $this->form->model($post)->saveRelationships();

        // Show appropriate notification
        Notification::make()
            ->success()
            ->title(match($this->action) {
                'draft' => 'Saved as Draft',
                default => 'Submitted for Review'
            })
            ->body(match($this->action) {
                'draft' => 'Article has been saved as draft successfully.',
                default => 'Article has been submitted for review.'
            })
            ->duration(5000)
            ->send();

        $this->redirect(route('dashboard.article.edit', ['uuid' => $post->uuid]), navigate: true);
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.article'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.article.create-article');
    }
}
