<?php

namespace App\Livewire\Dashboard\Article;

use App\Models\Blog\Post;
use App\Services\Blog\PostService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Article\ArticleForm;
use App\Enums\PostStatus;
use Filament\Notifications\Notification;

#[Layout('layouts.dashboard.dashboard-app')]
class EditArticle extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public Post $post;

    protected PostService $postService;

    public function boot(PostService $postService)
    {
        $this->postService = $postService;
    }

    public function mount(string $uuid): void
    {
        $this->post = $this->postService->findByUuid($uuid);

        if (!$this->post) {
            abort(404);
        }

        $this->form->fill([
            'post_title' => $this->post->post_title,
            'slug' => $this->post->slug,
            'post_excerpt' => $this->post->post_excerpt,
            'post_description' => $this->post->post_description,
            'post_status' => $this->post->post_status->value,
            'published_at' => $this->post->published_at,
            'created_by' => $this->post->created_by,
            'is_featured' => $this->post->is_featured,
            'categories' => $this->post->categories->pluck('id')->toArray(),
            'tags' => $this->post->tags->pluck('id')->toArray(),
            'featured_image' => $this->post->getMedia('featured_image')->first()?->id,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new ArticleForm())->getForm()
            )
            ->model($this->post ?? Post::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $formData = $this->form->getState();
        $formData['post_status'] = PostStatus::Draft->value;

        // Update post using PostService with UUID
        $this->postService->updateByUuid($this->post->uuid, $formData);

        $this->form->model($this->post)->saveRelationships();

        Notification::make()
            ->success()
            ->title('Article Updated')
            ->body('Your article has been saved as draft.')
            ->duration(5000)
            ->send();
    }

    public function save(): void
    {
        $formData = $this->form->getState();
        $formData['post_status'] = PostStatus::Pending->value;

        // Update post using PostService with UUID
        $this->postService->updateByUuid($this->post->uuid, $formData);

        $this->form->model($this->post)->saveRelationships();

        Notification::make()
            ->success()
            ->title('Article Updated')
            ->body('Your article has been updated and submitted for review.')
            ->duration(5000)
            ->send();
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.article'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.article.edit-article');
    }
}
