<?php

namespace App\Livewire\Dashboard\Classified;

use App\Enums\ClassifiedStatus;
use App\Enums\ClassifiedCondition;
use App\Enums\ClassifiedLabel;
use App\Models\Classified\Classified;
use App\Services\Classified\ClassifiedService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Classified\ClassifiedForm;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;

#[Layout('layouts.dashboard.dashboard-app')]
class CreateClassified extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public ?string $action = null;

    protected ClassifiedService $classifiedService;

    public function boot(ClassifiedService $classifiedService)
    {
        $this->classifiedService = $classifiedService;
    }
    public function mount(): void
    {
        $this->form->fill([
            'post_status' => ClassifiedStatus::Draft->value,
            'price' => 0.00,
            'discount_price' => 0.00,
            'condition' => ClassifiedCondition::Used->value,
            'label' => ClassifiedLabel::ForSale->value,
            'created_by' => auth()->guard('web')->user()?->id,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new ClassifiedForm())->getForm()
            )
            ->model(Classified::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $this->action = 'draft';
        $this->create();
    }

    public function create(): void
    {
        $data = $this->form->getState();
        $data['classified_status'] = match ($this->action) {
            'draft' => ClassifiedStatus::Draft->value,
            default => ClassifiedStatus::Pending->value
        };
        $data['user_id'] = auth()->guard('web')->user()?->id;

        try {
            DB::beginTransaction();

            // Create post using ClassifiedService
            $classified = $this->classifiedService->create($data);

            $this->form->model($classified)->saveRelationships();

            DB::commit();

            // Show appropriate notification
            Notification::make()
                ->success()
                ->title(match ($this->action) {
                    'draft' => 'Saved as Draft',
                    default => 'Submitted for Review'
                })
                ->body(match ($this->action) {
                    'draft' => 'Classified has been saved as draft successfully.',
                    default => 'Classified has been submitted for review.'
                })
                ->duration(5000)
                ->send();

            $this->redirect(route('dashboard.classified.edit', ['uuid' => $classified->uuid]), navigate: true);
        } catch (\Throwable $th) {
            DB::rollBack();
            Notification::make()
                ->danger()
                ->title('Error')
                ->body('An error occurred while creating the classified. Please try again.' . $th->getMessage())
                ->duration(5000)
                ->send();
        }
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.classified'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.classified.create-classified');
    }
}
