<?php

namespace App\Livewire\Dashboard\Classified;

use App\Models\Classified\Classified;
use App\Services\Classified\ClassifiedService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Classified\ClassifiedForm;
use Filament\Notifications\Notification;
use App\Enums\ClassifiedStatus;
use Illuminate\Support\Facades\DB;

#[Layout('layouts.dashboard.dashboard-app')]
class EditClassified extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public Classified $classified;

    protected ClassifiedService $classifiedService;

    public function boot(ClassifiedService $classifiedService)
    {
        $this->classifiedService = $classifiedService;
    }

    public function mount(string $uuid): void
    {
        $this->classified = $this->classifiedService->findByUuid($uuid);
        if (!$this->classified) {
            abort(404);
        }

        $this->form->fill([
            'title' => $this->classified->title,
            'slug' => $this->classified->slug,
            'description' => $this->classified->description,
            'price' => $this->classified->price,
            'discount_price' => $this->classified->discount_price,
            'discount_start' => $this->classified->discount_start,
            'discount_end' => $this->classified->discount_end,
            'key_features' => $this->classified->key_features,
            'maintenance_detail' => $this->classified->maintenance_detail,
            'website_link' => $this->classified->web_link,
            'featured' => $this->classified->is_featured,
            'location'  => $this->classified->location,
            'address'  => $this->classified->address,
            'city_id' =>  $this->classified->city_id,
            'state_id' => $this->classified->state_id,
            'country_id' =>  $this->classified->country_id,
            'country_name' => $this->classified->country?->country_name,
            'state_name' => $this->classified->state?->state_name,
            'city_name' => $this->classified->city?->city_name,
            'postal_code' => $this->classified->postal_code,
            'latitude'  => $this->classified->latitude,
            'longitude'  => $this->classified->longitude,
            'meta_title'  => $this->classified->meta_title,
            'meta_description'  => $this->classified->meta_description,
            'created_by' => $this->classified->created_by,
            'published_at' => $this->classified->published_at,
            'classified_status' => $this->classified->classified_status->value,
            'categories' => $this->classified->categories->pluck('id')->toArray(),
            'tags' => $this->classified->tags->pluck('id')->toArray(),
            'condition' => $this->classified->condition,
            'label' => $this->classified->label,
        ]);
    }
    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new ClassifiedForm())->getForm()
            )
            ->model($this->classified ?? Classified::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $formData = $this->form->getState();
        $formData['classified_status'] = ClassifiedStatus::Draft->value;

        // Update classified using ClassifiedService with UUID
        $this->classifiedService->updateByUuid($this->classified->uuid, $formData);

        $this->form->model($this->classified)->saveRelationships();

        Notification::make()
            ->success()
            ->title('Classified Updated')
            ->body('Your classified has been saved as draft.')
            ->duration(5000)
            ->send();
    }

    public function save(): void
    {
        $formData = $this->form->getState();
        $formData['classified_status'] = ClassifiedStatus::Pending->value;
        try {
            DB::beginTransaction();

            // Update classified using ClassifiedService with UUID
            $this->classifiedService->updateByUuid($this->classified->uuid, $formData);

            $this->form->model($this->classified)->saveRelationships();

            DB::commit();

            Notification::make()
                ->success()
                ->title('Classified Updated')
                ->body('Your classified has been updated and submitted for review.')
                ->duration(5000)
                ->send();
        } catch (\Exception $e) {
            DB::rollBack();

            Notification::make()
                ->danger()
                ->title('Error')
                ->body('An error occurred while updating the classified. Please try again.')
                ->duration(5000)
                ->send();

            report($e); // Log the error
        }

        return;
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.classified'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.classified.edit-classified');
    }
}
