<?php

namespace App\Livewire\Dashboard\Credits;

use App\Services\Credit\CreditPurchaseService;
use App\Services\Payment\StripePaymentService;
use App\Services\Payment\PayPalPaymentService;
use App\Notifications\CreditPurchaseSuccessful;
use App\Notifications\CreditPurchaseFailed;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Illuminate\Support\Facades\Auth;
use Filament\Notifications\Notification;

#[Layout('layouts.dashboard.dashboard-app')]
class CreditPurchase extends Component
{
    public int $quantity = 1;
    public string $selectedPaymentMethod = 'stripe';
    public array $pricingDetails = [];
    public array $availablePaymentMethods = [];
    public array $pricingTiers = [];
    public bool $isProcessing = false;
    public ?string $clientSecret = null;
    public ?string $stripePublishableKey = null;

    protected CreditPurchaseService $creditPurchaseService;
    protected StripePaymentService $stripePaymentService;
    protected PayPalPaymentService $payPalPaymentService;

    protected $rules = [
        'quantity' => 'required|integer|min:1',
        'selectedPaymentMethod' => 'required|string|in:stripe,paypal',
    ];

    public function boot(
        CreditPurchaseService $creditPurchaseService,
        StripePaymentService $stripePaymentService,
        PayPalPaymentService $payPalPaymentService
    ) {
        $this->creditPurchaseService = $creditPurchaseService;
        $this->stripePaymentService = $stripePaymentService;
        $this->payPalPaymentService = $payPalPaymentService;
    }

    public function mount()
    {
        $this->loadInitialData();
        $this->calculatePricing();
    }

    public function loadInitialData()
    {
        try {
            $this->availablePaymentMethods = $this->creditPurchaseService->getAvailablePaymentMethods();

            // Generate pricing tiers with calculated pricing for display
            $tierQuantities = [10, 25, 50, 100];
            $tierLabels = ['Small Pack', 'Medium Pack', 'Large Pack', 'Mega Pack'];

            $this->pricingTiers = [];
            foreach ($tierQuantities as $index => $quantity) {
                $pricing = $this->creditPurchaseService->calculateCreditPrice($quantity);
                $this->pricingTiers[] = [
                    'quantity' => $quantity,
                    'label' => $tierLabels[$index],
                    'price_per_credit' => $pricing['base_price'],
                    'total_price' => $pricing['total'],
                    'discount_percentage' => $pricing['discount_percentage'],
                    'savings' => $pricing['discount_amount'],
                ];
            }

            // Set default payment method to first available
            if (!empty($this->availablePaymentMethods)) {
                $this->selectedPaymentMethod = $this->availablePaymentMethods[0]['id'];
            }

            // Get Stripe config
            if (is_stripe_enabled()) {
                $this->stripePublishableKey = get_stripe_publishable_key();
            }
        } catch (\Exception $e) {
            Notification::make()
                ->title('Configuration Error')
                ->body('Unable to load payment configuration. Please contact support.')
                ->danger()
                ->persistent()
                ->send();
        }
    }

    public function updatedQuantity()
    {
        $this->calculatePricing();
        $this->resetErrorBag();
    }

    public function updatedSelectedPaymentMethod()
    {
        $this->resetErrorBag();
    }

    public function calculatePricing()
    {
        if ($this->quantity > 0) {
            try {
                $this->pricingDetails = $this->creditPurchaseService->calculateCreditPrice($this->quantity);
            } catch (\Exception $e) {
                Notification::make()
                    ->title('Pricing Error')
                    ->body('Unable to calculate pricing. Please refresh the page and try again.')
                    ->danger()
                    ->persistent()
                    ->send();

                // Reset to a safe state
                $this->pricingDetails = [];
            }
        }
    }

    public function selectPricingTier($quantity)
    {
        $this->quantity = $quantity;
        $this->calculatePricing();
    }

    public function processPayment()
    {
        $this->validate();

        // Validate quantity limits
        if (!$this->creditPurchaseService->validatePurchaseQuantity($this->quantity)) {
            $errorMessage = 'Invalid quantity selected. Please check the minimum and maximum limits.';

            Notification::make()
                ->title('Invalid Quantity')
                ->body($errorMessage)
                ->danger()
                ->persistent()
                ->send();

            return;
        }

        $this->isProcessing = true;

        try {
            $user = Auth::user();

            if ($this->selectedPaymentMethod === 'stripe') {
                $this->processStripePayment($user);
            } elseif ($this->selectedPaymentMethod === 'paypal') {
                $this->processPaypalPayment($user);
            }
        } catch (\Exception $e) {
            $errorMessage = 'An unexpected error occurred. Please try again.';

            Notification::make()
                ->title('Payment Processing Error')
                ->body($errorMessage)
                ->danger()
                ->persistent()
                ->send();

            $this->isProcessing = false;
        }
    }

    private function processStripePayment($user)
    {
        // Process credit purchase with Stripe
        $result = $this->creditPurchaseService->processCreditPurchase(
            $user,
            $this->quantity,
            'stripe',
            []
        );

        if (!$result['success']) {
            $errorMessage = $result['message'] ?? 'Stripe payment processing failed.';

            Notification::make()
                ->title('Payment Failed')
                ->body($errorMessage)
                ->danger()
                ->persistent()
                ->send();

            $this->isProcessing = false;
            return;
        }

        // If payment requires checkout (redirect to Stripe checkout)
        if (isset($result['requires_checkout']) && $result['requires_checkout']) {
            // Redirect to Stripe checkout page
            return redirect()->away($result['checkout_url']);
        } else {
            // Payment completed immediately (shouldn't happen with checkout flow)
            $this->isProcessing = false;

            Notification::make()
                ->title('Purchase Successful')
                ->body('Credit purchase completed successfully!')
                ->success()
                ->persistent()
                ->send();

            return redirect()->route('dashboard.account-subscription');
        }
    }

    private function processPaypalPayment($user)
    {
        // Process credit purchase with PayPal
        $result = $this->creditPurchaseService->processCreditPurchase(
            $user,
            $this->quantity,
            'paypal',
            []
        );

        if (!$result['success']) {
            $errorMessage = $result['message'] ?? 'PayPal payment processing failed.';

            Notification::make()
                ->title('Payment Failed')
                ->body($errorMessage)
                ->danger()
                ->persistent()
                ->send();

            $this->isProcessing = false;
            return;
        }

        // If payment requires checkout (redirect to PayPal checkout)
        if (isset($result['requires_checkout']) && $result['requires_checkout']) {
            // Redirect to PayPal checkout page
            return redirect()->away($result['checkout_url']);
        } else {
            // Payment completed immediately (shouldn't happen with normal flow)
            $this->isProcessing = false;

            Notification::make()
                ->title('Purchase Successful')
                ->body('Credit purchase completed successfully!')
                ->success()
                ->persistent()
                ->send();

            return redirect()->route('dashboard.account-subscription');
        }
    }

    public function handlePaymentSuccess(CreditPurchaseService $creditPurchaseService)
    {
        // Handle successful payment return from Stripe Checkout
        $user = Auth::user();
        $sessionId = request('session_id');
        
        if ($sessionId) {
            // Process the checkout session, verify payment, and add credits if successful
            $result = $creditPurchaseService->handleCheckoutSessionCompleted($sessionId);
            
            if ($result['success']) {
                Notification::make()
                    ->title('Purchase Successful')
                    ->body('Your credit purchase has been processed and credits have been added to your account!')
                    ->success()
                    ->persistent()
                    ->send();
            } else {
                Notification::make()
                    ->title('Payment Issue')
                    ->body($result['message'] ?? 'There was an issue processing your payment. Please contact support.')
                    ->warning()
                    ->persistent()
                    ->send();
            }
        } else {
            Notification::make()
                ->title('Payment Processing')
                ->body('Your payment is being processed. Credits will be added to your account once the payment is confirmed.')
                ->info()
                ->persistent()
                ->send();
        }

        // Redirect to account subscription page
        return redirect()->route('dashboard.account-subscription');
    }

    public function handlePaymentCancellation()
    {
        // Handle cancelled payment from Stripe Checkout
        $user = Auth::user();

        Notification::make()
            ->title('Payment Cancelled')
            ->body('Your payment was cancelled. You can try again anytime.')
            ->warning()
            ->persistent()
            ->send();

        // Reset form state
        $this->resetForm();
    }

    public function handlePayPalSuccess(CreditPurchaseService $creditPurchaseService)
    {
        // Handle successful payment return from PayPal
        $user = Auth::user();
        $orderId = session('paypal_order_id');

        if ($orderId) {
            // Capture the PayPal payment and add credits only if payment is successful
            $captureResult = $creditPurchaseService->handlePayPalPaymentSuccess($orderId);
            
            // Clear the session to prevent reprocessing
            session()->forget('paypal_order_id');

            if ($captureResult['success']) {
                Notification::make()
                    ->title('Purchase Successful')
                    ->body('Your PayPal payment has been processed and credits have been added to your account successfully!')
                    ->success()
                    ->persistent()
                    ->send();
            } else {
                Notification::make()
                    ->title('Payment Error')
                    ->body($captureResult['message'] ?? 'Unable to complete payment. Please contact support.')
                    ->danger()
                    ->persistent()
                    ->send();
            }
        }

        // Redirect to account subscription page
        return redirect()->route('dashboard.account-subscription');
    }

    public function handlePayPalCancellation()
    {
        // Handle cancelled payment from PayPal
        $user = Auth::user();

        Notification::make()
            ->title('PayPal Payment Cancelled')
            ->body('Your PayPal payment was cancelled. You can try again anytime.')
            ->warning()
            ->persistent()
            ->send();

        // Reset form state
        $this->resetForm();

        return redirect()->route('dashboard.credits.purchase');
    }

    public function resetForm()
    {
        $this->quantity = 1;
        $this->clientSecret = null;
        $this->isProcessing = false;
        $this->calculatePricing();
        $this->resetErrorBag();
    }

    public function render()
    {
        return view('livewire.dashboard.credits.credit-purchase');
    }
}
