<?php

namespace App\Livewire\Dashboard\Equipment;

use Livewire\Component;
use App\Enums\EquipmentStatus;
use App\Models\Equipment\Equipment;
use App\Services\Equipment\EquipmentService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Attributes\Layout;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;
use App\Filament\Dashboard\Equipment\EquipmentForm;
use Illuminate\Support\Facades\DB;
use Throwable;

#[Layout('layouts.dashboard.dashboard-app')]
class CreateEquipment extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public ?string $action = null;

    protected EquipmentService $equipmentService;

    public function boot(EquipmentService $equipmentService)
    {
        $this->equipmentService = $equipmentService;
    }

    public function mount(): void
    {
        $this->form->fill([
            'status' => EquipmentStatus::Draft->value,
            'user_id' => Auth::id(),
            'created_by' => Auth::id(),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new EquipmentForm())->getForm(),
            )
            ->model(Equipment::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $this->action = 'draft';
        $this->create();
    }

    public function create(): void
    {
        $data = $this->form->getState();
        try {
            DB::beginTransaction();
            $equipment = $this->equipmentService->create($data);
            if ($equipment) {
                $this->form->model($equipment)->saveRelationships();
            }
            DB::commit();
            Notification::make()
                ->success()
                ->title(match ($this->action) {
                    'draft' => 'Saved as Draft',
                    default => 'Submitted for Review'
                })
                ->body(match ($this->action) {
                    'draft' => 'Equipment has been saved as draft successfully.',
                    default => 'Equipment has been submitted for review.'
                })
                ->duration(5000)
                ->send();
            $this->redirect(route('dashboard.equipments.edit', ['uuid' => $equipment->uuid]), navigate: true);
        } catch (Throwable $th) {
            DB::rollBack();
            Notification::make()
                ->danger()
                ->title('Error')
                ->body('An error occurred while creating the classified. Please try again.' . $th->getMessage())
                ->duration(5000)
                ->send();
        }
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.equipments'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.equipment.create-equipment');
    }
}
