<?php

namespace App\Livewire\Dashboard\Equipment;

use App\Enums\EquipmentStatus;
use Livewire\Component;
use App\Models\Equipment\Equipment;
use App\Services\Equipment\EquipmentService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Attributes\Layout;
use Filament\Notifications\Notification;
use App\Filament\Dashboard\Equipment\EquipmentForm;
use Illuminate\Support\Facades\DB;
use Throwable;

#[Layout('layouts.dashboard.dashboard-app')]
class EditEquipment extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public Equipment $equipment;
    public ?string $action = null;


    protected EquipmentService $equipmentService;

    public function boot(EquipmentService $equipmentService)
    {
        $this->equipmentService = $equipmentService;
    }

    public function mount(string $uuid): void
    {
        $this->equipment = $this->equipmentService->findByUuid($uuid);

        if (!$this->equipment) {
            abort(404);
        }

        $this->form->fill([
            'title' => $this->equipment->title,
            'slug' => $this->equipment->slug,
            'description' => $this->equipment->description,
            'user_id' => $this->equipment->user_id,
            'status' => $this->equipment->status,
            'tags' => $this->equipment->tags,
            'address' => $this->equipment->address,
            'city_id' => $this->equipment->city_id,
            'state_id' => $this->equipment->state_id,
            'country_id' => $this->equipment->country_id,
            'specification' => $this->equipment->specification,
            'use_case' => $this->equipment->use_case,
            'constraint' => $this->equipment->constraint,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new EquipmentForm())->getForm(),
            )
            ->model($this->equipment ?? Equipment::class)
            ->statePath('data');
    }


    public function update(): void
    {
        $data = $this->form->getState();
        try {
            DB::beginTransaction();
            $equipment = $this->equipmentService->updateByUuid($this->equipment->uuid, $data);
            if ($equipment) {
                $this->form->model($equipment)->saveRelationships();
            }
            DB::commit();
            Notification::make()
                ->success()
                ->title('Equipment Updated')
                ->body('Equipment has been updated successfully.')
                ->duration(5000)
                ->send();
        } catch (Throwable $th) {
            DB::rollBack();
            Notification::make()
                ->danger()
                ->title('Error')
                ->body('An error occurred while updating the equipment. Please try again.' . $th->getMessage())
                ->duration(5000)
                ->send();
        }
        $equipment = $this->equipmentService->updateByUuid($this->equipment->uuid, $data);
    }


    public function saveAsDraft(): void
    {
        $formData = $this->form->getState();
        $formData['status'] = EquipmentStatus::Draft->value;

        // Update classified using ClassifiedService with UUID
        $this->equipmentService->updateByUuid($this->equipment->uuid, $formData);

        $this->form->model($this->equipment)->saveRelationships();

        Notification::make()
            ->success()
            ->title('Equipment Updated')
            ->body('Your equipment has been saved as draft.')
            ->duration(5000)
            ->send();
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.equipments'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.equipment.edit-equipment');
    }
}
