<?php

namespace App\Livewire\Dashboard\Job;

use Livewire\Component;
use App\Enums\JobStatus;
use App\Models\Job\Job;
use App\Services\Job\JobService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Job\JobForm;
use Filament\Notifications\Notification;

#[Layout('layouts.dashboard.dashboard-app')]
class CreateJob extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public ?string $action = null;

    protected JobService $jobService;

    public function boot(JobService $jobService)
    {
        $this->jobService = $jobService;
    }

    public function mount(): void
    {
        $this->form->fill([
            'job_status' => JobStatus::Draft->value,
            'created_by' => auth()->guard('web')->user()?->id,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new JobForm())->getForm()
            )
            ->model(Job::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $this->action = 'draft';
        $this->create();   
    }


    public function create(): void
    {
        $data = $this->form->getState();
        // dd($data);
        $data['job_status'] = match($this->action) {
            'draft' => JobStatus::Draft->value,
            default => JobStatus::Pending->value
        };
        $data['user_id'] = auth()->guard('web')->user()?->id;

        // Create job using JobService
        $job = $this->jobService->create($data);
        $this->form->model($job)->saveRelationships();

        // Show appropriate notification
        Notification::make()
            ->success()
            ->title(match($this->action) {
                'draft' => 'Saved as Draft',
                default => 'Submitted for Review'
            })
            ->body(match($this->action) {
                'draft' => 'Job has been saved as draft successfully.',
                default => 'Job has been submitted for review.'
            })
            ->duration(5000)
            ->send();

        $this->redirect(route('dashboard.job.edit', ['uuid' => $job->uuid]), navigate: true);
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.job'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.job.create-job');
    }
}
