<?php

namespace App\Livewire\Dashboard\Job;

use App\Enums\JobStatus;
use Livewire\Component;
use App\Models\Job\Job;
use App\Services\Job\JobService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Job\JobForm;
use Filament\Notifications\Notification;

#[Layout('layouts.dashboard.dashboard-app')]
class EditJob extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public Job $job;

    protected JobService $jobService;

    public function boot(JobService $jobService)
    {
        $this->jobService = $jobService;
    }

    public function mount(string $uuid): void
    {
        $this->job = $this->jobService->findByUuid($uuid);


        if (!$this->job) {
            abort(404);
        }

        $this->form->fill([

            'job_title' => $this->job->job_title,
            'slug' => $this->job->slug,
            'job_description' => $this->job->job_description,
            'user_id' => $this->job->user_id,
            'job_role_id' => $this->job->job_role_id,
            'job_industry_id' => $this->job->job_industry_id,
            'available_positions' => $this->job->available_positions,
            'preferred_gender' => $this->job->preferred_gender,
            'min_salary' => $this->job->min_salary,
            'max_salary' => $this->job->max_salary,
            'currency_id' => $this->job->currency_id,
            'salary_period' => $this->job->salary_period,
            'job_shift' => $this->job->job_shift,
            'work_mode' => $this->job->work_mode,
            'apply_by_date' => $this->job->apply_by_date,
            'deactivate_apply_by' => $this->job->deactivate_apply_by,
            'key_responsibilities' => $this->job->key_responsibilities,
            'required_qualifications' => $this->job->required_qualifications,
            'nice_to_have_skills' => $this->job->nice_to_have_skills,
            'bonus_and_benefits' => $this->job->bonus_and_benefits,
            'country_id' => $this->job->country_id,
            'state_id' => $this->job->state_id,
            'city_id' => $this->job->city_id,
            'is_featured' => $this->job->is_featured,
            'is_active' => $this->job->is_active,
            'created_by' => $this->job->created_by,
            'updated_by' => $this->job->updated_by,
            'published_at' => $this->job->published_at,
            'job_status' => $this->job->job_status,
            'expired_at' => $this->job->expired_at,
        ]);
    }
    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new JobForm())->getForm()
            )
            ->model($this->job ?? Job::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $formData = $this->form->getState();
        $formData['job_status'] = JobStatus::Draft->value;

        // Update job using JobService with UUID
        $this->jobService->updateByUuid($this->job->uuid, $formData);

        $this->form->model($this->job)->saveRelationships();

        Notification::make()
            ->success()
            ->title('Job Updated')
            ->body('Your job has been saved as draft.')
            ->duration(5000)
            ->send();
    }

    public function save(): void
    {
        $formData = $this->form->getState();
        $formData['job_status'] = JobStatus::Pending->value;
        // Update job using JobService with UUID
        $this->jobService->updateByUuid($this->job->uuid, $formData);

        $this->form->model($this->job)->saveRelationships();

        Notification::make()
            ->success()
            ->title('Job Updated')
            ->body('Your job has been updated and submitted for review.')
            ->duration(5000)
            ->send();
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.job'), navigate: true);
    }


    public function render()
    {
        return view('livewire.dashboard.job.edit-job');
    }
}
