<?php

namespace App\Livewire\Dashboard\ManageLead;

use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Models\Lead\Lead;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Grid;
use App\Enums\LeadStatus;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Actions\Action;
use Filament\Notifications\Notification;

#[Layout('layouts.dashboard.dashboard-app')]
class EditLead extends Component implements HasForms, HasActions
{
    use InteractsWithForms;
    use InteractsWithActions;

    public Lead $lead;
    public ?array $data = [];

    public function mount(string $uuid): void
    {
        $this->lead = Lead::where('uuid', $uuid)->firstOrFail();
        $this->form->fill([
            'first_name' => $this->lead->first_name,
            'last_name' => $this->lead->last_name,
            'email' => $this->lead->email,
            'phone' => $this->lead->phone,
            'message' => $this->lead->message,
            'lead_status' => $this->lead->lead_status?->value ?? 'pending',
            'lead_source' => $this->lead->lead_source,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Contact Information')
                    ->schema([
                        Grid::make(2)->schema([
                            TextInput::make('first_name')
                                ->label('First Name')
                                ->required()
                                ->maxLength(255),
                            
                            TextInput::make('last_name')
                                ->label('Last Name')
                                ->required()
                                ->maxLength(255),
                        ]),
                        
                        Grid::make(2)->schema([
                            TextInput::make('email')
                                ->label('Email')
                                ->email()
                                ->required()
                                ->maxLength(255),
                            
                            TextInput::make('phone')
                                ->label('Phone')
                                ->tel()
                                ->maxLength(255),
                        ]),
                    ]),

                Section::make('Lead Details')
                    ->schema([
                        Select::make('lead_status')
                            ->label('Lead Status')
                            ->options([
                                'pending' => 'Pending',
                                'accepted' => 'Accepted',
                                'rejected' => 'Rejected',
                                'in_progress' => 'In Progress',
                                'completed' => 'Completed',
                            ])
                            ->default('pending')
                            ->required(),
                            
                        TextInput::make('lead_source')
                            ->label('Lead Source')
                            ->maxLength(255),
                            
                        Textarea::make('message')
                            ->label('Message')
                            ->rows(4)
                            ->columnSpanFull(),
                    ]),
            ])
            ->statePath('data');
    }

    public function save(): void
    {
        $data = $this->form->getState();
        
        $this->lead->update([
            'first_name' => $data['first_name'],
            'last_name' => $data['last_name'],
            'email' => $data['email'],
            'phone' => $data['phone'],
            'message' => $data['message'],
            'lead_status' => $data['lead_status'],
            'lead_source' => $data['lead_source'],
        ]);

        Notification::make()
            ->title('Lead updated successfully!')
            ->success()
            ->send();

        $this->redirect(route('dashboard.manage-lead'));
    }

    public function cancelAction(): Action
    {
        return Action::make('cancel')
            ->label('Cancel')
            ->color('gray')
            ->url(route('dashboard.manage-lead'));
    }

    public function render()
    {
        return view('livewire.dashboard.manage-lead.edit-lead');
    }
} 