<?php

namespace App\Livewire\Dashboard\ManageLead;

use Livewire\Component;
use Livewire\Attributes\Layout;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Actions\EditAction;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use App\Models\Lead\Lead;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\BulkAction;
use Filament\Tables\Actions\BulkActionGroup;
use App\Services\Lead\LeadService;
use Illuminate\Support\Facades\Auth;
#[Layout('layouts.dashboard.dashboard-app')]
class ManageLead extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;
    protected $leadService;

    public function boot(LeadService $leadService)
    {
        $this->leadService = $leadService;
    }


    public function getAllLeads()
    {
        $args = [
            'user_id' => Auth::id(),
        ];

        $leads = $this->leadService->getAll($args);
        
        return $leads;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getAllLeads())
            ->columns([
                TextColumn::make('leadForm.title')
                    ->label('Form Name')
                    ->searchable()
                    ->sortable()
                    ->default('N/A'),
                
                TextColumn::make('full_name')
                    ->label('Name')
                    ->getStateUsing(function (Lead $record): string {
                        return trim($record->first_name . ' ' . $record->last_name) ?: 'N/A';
                    })
                    ->searchable(['first_name', 'last_name'])
                    ->sortable(),
                
                TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->default('N/A'),
                
                TextColumn::make('phone')
                    ->label('Phone')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->default('N/A'),
                TextColumn::make('lead_status')
                    ->label('Status')
                    ->badge()
                    ->getStateUsing(function (Lead $record): string {
                        return $record->lead_status->getLabel() ?: 'N/A';
                    })
                    ->searchable()
                    ->sortable()
                    ->default('N/A'),
                TextColumn::make('created_at')
                    ->label('Created Date')
                    ->dateTime('M j, Y g:i A')
                    ->sortable()
                    ->toggleable(),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('view')
                        ->label('View')
                        ->icon('heroicon-o-eye')
                        ->color('info')
                        ->url(fn (Lead $record): string => route('dashboard.leads.view', $record->uuid))
                        ->openUrlInNewTab(false)
                        ->visible(fn (Lead $record):string=> Auth::user()->can('view',['App\Models\Lead\Lead',$record->uuid])),
                ])
                ->label('Actions')
                ->icon('heroicon-o-ellipsis-vertical')
                ->size('sm')
                ->color('gray')
            ])
            ->filters([
                // Add filters here if needed
            ])
            ->bulkActions([
               BulkActionGroup::make([
                DeleteBulkAction::make('delete')
                    ->label('Delete')
                    ->icon('heroicon-o-trash')
                    ->color('danger')
                    ->action(fn (Collection $records) => $records->each(fn (Lead $record) => $record->delete())),
               ])
               
            ])
            ->defaultSort('created_at', 'desc')
            ->paginated([10, 25, 50, 100])
            ->searchable()
            ->emptyStateHeading('No leads found')
            ->emptyStateDescription('No leads have been created yet.')
            ->emptyStateIcon('heroicon-o-inbox');
    }

    public function render()
    {
        return view('livewire.dashboard.manage-lead.manage-lead');
    }
}
