<?php

namespace App\Livewire\Dashboard\ManageLead;

use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use Livewire\Component;
use App\Services\Lead\LeadService;
use Livewire\Attributes\Layout;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Actions\Action as TableAction;
use Filament\Forms\Components\Select;
use App\Models\User\User;
use App\Enums\LeadUserStatus;
use Filament\Notifications\Notification;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables\Actions\BulkAction;
use Filament\Tables\Actions\BulkActionGroup;
use Illuminate\Database\Eloquent\Collection;

#[Layout('layouts.dashboard.dashboard-app')]
class ViewLead extends Component implements HasTable, HasForms
{
    use InteractsWithTable, InteractsWithForms;

    protected $leadService;

    public $lead;
    public $activeTab = 'assigned';

    public function boot(LeadService $leadService)
    {
        $this->leadService = $leadService;
    }

    public function switchTab(string $tab): void
    {
        $this->activeTab = $tab;
    }

    public function setActiveTab($tab)
    {
        $this->activeTab = $tab;
        // Reset table when switching tabs to ensure fresh data
        $this->resetTable();
    }

    public function getAvailableUsersCount(): int
    {
        return $this->getAvailableUsers()->count();
    }

    public function getAssignedUsersCount(): int
    {
        return $this->lead->users()->count();
    }

    public function mount(string $uuid)
    {
        
        $this->lead = $this->leadService->findByUuid($uuid);
        
        // Load the users relationship with pivot data
        $this->lead->load([
            'users' => function ($query) {
                $query->withPivot(['lead_status']);
            },
            'leadForm.memberCategories'
        ]);
    }

    public function updateUserStatus($userId, $newStatus)
    {
        $this->lead->users()->updateExistingPivot($userId, [
            'lead_status' => $newStatus
        ]);

        // Refresh record data
        $this->refreshRecordData();

        // Reset table to refresh the view
        $this->resetTable();

        Notification::make()
            ->title('User status updated successfully')
            ->success()
            ->send();
    }

    public function detachUser($userId)
    {
        $user = User::find($userId);

        if (!$user) {
            Notification::make()
                ->title('User not found')
                ->danger()
                ->send();
            return;
        }

        // Detach the user
        $this->lead->users()->detach($userId);

        // Refresh record data
        $this->refreshRecordData();

        // Reset table to refresh both views
        $this->resetTable();

        Notification::make()
            ->title("User '{$user->name}' removed from lead")
            ->success()
            ->send();
    }

    public function assignUser($userId, $status)
    {
        // Validate the user ID
        if (!$userId) {
            Notification::make()
                ->title('Please select a user')
                ->danger()
                ->send();
            return;
        }

        // Check if user is already assigned
        if ($this->lead->users()->where('user_id', $userId)->exists()) {
            Notification::make()
                ->title('User is already assigned to this lead')
                ->warning()
                ->send();
            return;
        }

        $user = User::find($userId);

        if (!$user) {
            Notification::make()
                ->title('User not found')
                ->danger()
                ->send();
            return;
        }

        // Attach the user with the specified status
        $this->lead->users()->attach($userId, [
            'lead_status' => $status,
        ]);

        // Refresh record data
        $this->refreshRecordData();

        // Reset table to refresh both views
        $this->resetTable();

        Notification::make()
            ->title("User '{$user->name}' assigned successfully")
            ->success()
            ->send();

        // Switch to assigned tab to show the newly assigned user
        $this->activeTab = 'assigned';
    }

    public function getAssignedUsers(): Builder
    {
        return $this->lead->users()->withPivot(['lead_status'])->getQuery();
    }

    public function getAvailableUsers(): Builder
    {
        // Get the member categories associated with this lead's form
        $leadFormCategoryIds = $this->lead->leadForm?->memberCategories?->pluck('id')?->toArray() ?? [];

        if (empty($leadFormCategoryIds)) {
            // If no categories are set for the lead form, return empty query
            return User::query()->whereRaw('1 = 0');
        }

        // Get users who belong to these categories but are not already assigned to this lead
        $assignedUserIds = $this->lead->users->pluck('id')->toArray();

        return User::query()
            ->whereHas('memberCategories', function ($query) use ($leadFormCategoryIds) {
                $query->whereIn('member_categories.id', $leadFormCategoryIds);
            })
            ->where('users.id', '!=', $this->lead->created_by) // Exclude lead creator
            ->whereNotIn('users.id', $assignedUserIds) // Exclude already assigned users
            ->active(); // Only active users
    }

    protected function refreshRecordData()
    {
        // Reload the record with fresh relationship data
        $this->lead->load([
            'users' => function ($query) {
                $query->withPivot(['lead_status']);
            },
            'leadForm.memberCategories'
        ]);
    }

    public function table(Table $table): Table
    {
        $table = $this->getAssignedUsersTable($table);

        if ($this->activeTab === 'available') {
            $table = $this->getAvailableUsersTable($table);
        }
        return $table;
    }

    public function getLeadsQuery(): Builder
    {
        return $this->lead->users()->getQuery();
    }

    protected function getAssignedUsersTable(Table $table): Table
    {
        return $table
            ->query($this->getAssignedUsers())
            ->heading('Assigned Users')
            ->description('Users currently assigned to this lead')
            ->columns([
                SpatieMediaLibraryImageColumn::make('avatar')
                    ->collection('avatar')
                    ->conversion('small')
                    ->label('Avatar')
                    ->circular()
                    ->size(40),

                TextColumn::make('name')
                    ->label('Name')
                    ->searchable()
                    ->sortable()
                    ->weight('medium')
                    ->color('primary'),

                TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->copyMessage('Email address copied')
                    ->copyMessageDuration(1500)
                    ->icon('heroicon-m-envelope'),

                TextColumn::make('phone')
                    ->label('Phone')
                    ->searchable()
                    ->getStateUsing(function ($record) {
                        return $record->profile?->phone ?? 'N/A';
                    }),

                TextColumn::make('lead_status')
                    ->label('Status')
                    ->badge()
                    ->getStateUsing(function ($record) {
                        $status = $record->lead_status ?? 'N/A';
                        return LeadUserStatus::from($status)->getLabel();
                    }),

            ])->bulkActions([
                BulkActionGroup::make([
                    BulkAction::make('updateStatusBulk')
                        ->label('Update Status')
                        ->icon('heroicon-o-pencil-square')
                        ->color('warning')
                        ->modalHeading('Update Status for Selected Users')
                        ->modalDescription('Change the status for all selected users.')
                        ->form([
                            Select::make('lead_status')
                                ->label('New Status')
                                ->options(LeadUserStatus::class)
                                ->required()
                        ])
                        ->action(function (Collection $records, array $data): void {
                            foreach ($records as $record) {
                                $this->lead->users()->updateExistingPivot($record->id, [
                                    'lead_status' => $data['lead_status']
                                ]);
                            }

                            // Refresh record data
                            $this->refreshRecordData();

                            // Reset table to refresh
                            $this->resetTable();

                            Notification::make()
                                ->title('Status updated for ' . $records->count() . ' user(s)')
                                ->success()
                                ->send();
                        })
                        ->deselectRecordsAfterCompletion(),

                    BulkAction::make('removeBulk')
                        ->label('Remove Selected')
                        ->icon('heroicon-o-trash')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->modalHeading('Detach User Assignments')
                        ->modalDescription('Are you sure you want to detach the selected users from this lead? This action cannot be undone.')
                        ->action(function (Collection $records): void {
                            foreach ($records as $record) {
                                $this->lead->users()->detach($record->id);
                            }

                            // Refresh record data
                            $this->refreshRecordData();

                            // Reset table to refresh both views
                            $this->resetTable();

                            Notification::make()
                                ->title('Detached ' . $records->count() . ' user(s) from lead')
                                ->success()
                                ->send();
                        })
                        ->deselectRecordsAfterCompletion(),
                ])
            ])
            ->emptyStateHeading('No Users Assigned')
            ->emptyStateDescription('This lead doesn\'t have any assigned users yet.')
            ->emptyStateIcon('heroicon-o-users')
            ->emptyStateActions([
                TableAction::make('assignFirstUser')
                    ->label('Assign First User')
                    ->icon('heroicon-o-user-plus')
                    ->color('primary')
                    ->modalHeading('Assign User to Lead')
                    ->modalDescription('Select a user to assign to this lead and set their initial status.')
                    ->form([
                        Select::make('user_id')
                            ->label('Select User')
                            ->options(function () {
                                return User::active()
                                    ->whereNotIn('id', $this->lead->users->pluck('id'))
                                    ->get()
                                    ->mapWithKeys(function ($user) {
                                        return [$user->id => "{$user->name} ({$user->email})"];
                                    });
                            })
                            ->searchable()
                            ->required()
                            ->helperText('Select a user to assign to this lead.'),

                        Select::make('lead_status')
                            ->label('Initial Status')
                            ->options(LeadUserStatus::class)
                            ->default(LeadUserStatus::New->value)
                            ->required()
                            ->helperText('Set the initial status for this user\'s assignment to the lead.'),
                    ])
                    ->action(function (array $data) {
                        $this->assignUser($data['user_id'], $data['lead_status']);
                    })
                    ->modalSubmitActionLabel('Assign User'),
            ])
            ->striped()
            ->paginated([10, 25, 50, 100])
            ->defaultSort('name', 'asc');
    }

    protected function getAvailableUsersTable(Table $table): Table
    {
        return $table
            ->query($this->getAvailableUsers())
            ->heading('Available Users')
            ->description('Users with matching member categories who can be assigned to this lead')
            ->columns([
                SpatieMediaLibraryImageColumn::make('avatar')
                    ->collection('avatar')
                    ->conversion('small')
                    ->label('Avatar')
                    ->circular()
                    ->size(40),

                TextColumn::make('name')
                    ->label('Name')
                    ->searchable()
                    ->sortable()
                    ->weight('medium')
                    ->color('primary'),

                TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->copyMessage('Email address copied')
                    ->copyMessageDuration(1500)
                    ->icon('heroicon-m-envelope'),

                TextColumn::make('phone')
                    ->label('Phone')
                    ->searchable()
                    ->getStateUsing(function ($record) {
                        return $record->profile?->phone ?? 'N/A';
                    }),

                TextColumn::make('member_categories')
                    ->label('Categories')
                    ->getStateUsing(function ($record) {
                        $leadFormCategoryIds = $this->lead->leadForm?->memberCategories?->pluck('id')?->toArray() ?? [];
                        $matchingCategories = $record->memberCategories()
                            ->whereIn('member_categories.id', $leadFormCategoryIds)
                            ->pluck('category_name')
                            ->toArray();

                        if (empty($matchingCategories)) {
                            return 'No matching categories';
                        }

                        // Create HTML with badges and line breaks
                        $badges = array_map(function ($category) {
                            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200">' . htmlspecialchars($category) . '</span>';
                        }, $matchingCategories);

                        return implode('<br>', $badges);
                    })
                    ->html()
                    ->wrap(),

                TextColumn::make('profile.status')
                    ->label('Status')
                    ->badge()
                    ->getStateUsing(function ($record) {
                        return $record->profile?->status ?? 'N/A';
                    }),
            ])
            ->actions([
                TableAction::make('assign')
                    ->label('Assign')
                    ->icon('heroicon-o-user-plus')
                    ->color('primary')
                    ->modalHeading('Assign User to Lead')
                    ->modalDescription('Set the initial status for this user\'s assignment to the lead.')
                    ->form([
                        Select::make('lead_status')
                            ->label('Initial Status')
                            ->options(LeadUserStatus::class)
                            ->default(LeadUserStatus::New->value)
                            ->required()
                            ->helperText('Set the initial status for this user\'s assignment to the lead.'),
                    ])
                    ->action(function (User $record, array $data): void {
                        $this->assignUser($record->id, $data['lead_status']);
                    })
                    ->modalSubmitActionLabel('Assign User'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    BulkAction::make('assignBulk')
                        ->label('Assign Selected')
                        ->icon('heroicon-o-user-plus')
                        ->color('primary')
                        ->modalHeading('Assign Selected Users to Lead')
                        ->modalDescription('Set the initial status for all selected users.')
                        ->form([
                            Select::make('lead_status')
                                ->label('Initial Status')
                                ->options(LeadUserStatus::class)
                                ->default(LeadUserStatus::New->value)
                                ->required()
                        ])
                        ->action(function (Collection $records, array $data): void {
                            $successCount = 0;
                            foreach ($records as $record) {
                                // Check if user is not already assigned
                                if (!$this->lead->users()->where('user_id', $record->id)->exists()) {
                                    $this->lead->users()->attach($record->id, [
                                        'lead_status' => $data['lead_status'],
                                    ]);
                                    $successCount++;
                                }
                            }

                            // Refresh record data
                            $this->refreshRecordData();

                            // Reset table to refresh both views
                            $this->resetTable();

                            if ($successCount > 0) {
                                // Switch to assigned tab to show newly assigned users
                                $this->activeTab = 'assigned';

                                Notification::make()
                                    ->title("Assigned {$successCount} user(s) to lead")
                                    ->success()
                                    ->send();
                            } else {
                                Notification::make()
                                    ->title('No users were assigned (they may already be assigned)')
                                    ->warning()
                                    ->send();
                            }
                        })
                        ->deselectRecordsAfterCompletion(),
                ])
            ])
            ->emptyStateHeading('No Available Users')
            ->emptyStateDescription('No users with matching member categories are available for assignment, or this lead form has no associated categories.')
            ->emptyStateIcon('heroicon-o-user-group')
            ->striped()
            ->paginated([10, 25, 50, 100, 'all'])
            ->defaultPaginationPageOption(25)
            ->defaultSort('name', 'asc');
    }

    public function render()
    {
        return view('livewire.dashboard.manage-lead.view-lead');
    }
}
